<?php
/**
 * Raptor Utility Functions
 */

defined( 'ABSPATH' ) || exit;


/**  
 *  Output a file uri from within the branding directory
 *
 *  @since 1.0
 *  @param string $file The file name of the requested file
 */
function raptor_get_branding_url( string $file = '' ) {
    echo esc_url( get_template_directory_uri() . '/assets/branding/' . $file );
}


/**  
 *  Output a file uri from within the media directory
 *
 *  @since 1.0
 *  @param string $file The file name of the requested file
 *  @param string $media_dir The directory name where the file is expected
 */
function raptor_get_asset_url__old( string $file = '', string $asset_dir = 'images' ) {
    echo esc_url( get_template_directory_uri() . '/assets/' . $asset_dir . '/' . $file );
}


/**  
 *  Return a Raptor setting
 *
 *  @since 1.0
 *  @param string $key The suffix of the setting name
 *  @return mixed
 */
function raptor_get_setting( string $key = '' ) {
    return get_option( 'raptor_' . $key );
}


/**  
 *  Convert a string with hyphans to spaces and capitalize
 *
 *  @since 1.0
 *  @param string $str The string to convert
 *  @return string
 */
function raptor_to_nicename( string $str = '' ) {
    $nicename = str_replace( '-', ' ', $str );
    $nicename = str_replace( '_', ' ', $str );
    $nicename = ucwords( $nicename );

    return $nicename;
}


/**  
 *  Prep the JS object for wp_localize_script
 *
 *  @return array
 */
function raptor_js_object() {
    $cookies = raptor_get_settings( 'raptor', 'cookies' );
    $form_settings = raptor_get_settings( 'forms' );

    $form = [];

    if ( !empty( $form_settings ) && isset( $form_settings[ raptor_acf_get_field_selector( 'success_message' ) ] ) ) {
        $form = [
            'successHeading'    => $form_settings[ raptor_acf_get_field_selector( 'success_message' ) ]['heading'],
            'successBody'       => $form_settings[ raptor_acf_get_field_selector( 'success_message' ) ]['body'],
            'errorHeading'      => $form_settings[ raptor_acf_get_field_selector( 'error_message' ) ]['heading'],
            'errorBody'         => $form_settings[ raptor_acf_get_field_selector( 'error_message' ) ]['body'],
            'redirectsEnabled'  => apply_filters( 'raptor/forms/redirects_enabled', (bool) raptor_frontend_version_compare( null, '>=' ) ),
            'recaptchaSiteKey'  => $form_settings['recaptcha'][ raptor_acf_get_field_selector( 'site_key' ) ],
            'conditionalLogicEnabled'  => apply_filters(
                'raptor/forms/conditional_logic_enabled',
                raptor_frontend_version_compare( '1.5', '>=' )
            ),
        ];

        $form = apply_filters( 'raptor/forms/settings', $form );
    }

    return apply_filters(
        'raptor_js_object',
        [
            'name'      => get_bloginfo( 'name' ),
            'url'       => get_option( 'home' ),
            'theme'     => get_template_directory_uri(),
            'ajaxUrl'   => admin_url( 'admin-ajax.php' ),
            'form'      => $form,
            'cookie'    => [
                'message'       => apply_filters( 'the_content', $cookies ? $cookies['message'] : '' ),
                'policyLink'    => get_privacy_policy_url()
            ]
        ]
    );
}


/**
 * Convert hexidecimal value to rgb.
 * 
 * @param string $hex
 * @return string
 */
function raptor_hex_to_rgb( string $hex ) {
    $hex      = str_replace('#', '', $hex);
    $length   = strlen($hex);
    $rgb['r'] = hexdec($length == 6 ? substr($hex, 0, 2) : ($length == 3 ? str_repeat(substr($hex, 0, 1), 2) : 0));
    $rgb['g'] = hexdec($length == 6 ? substr($hex, 2, 2) : ($length == 3 ? str_repeat(substr($hex, 1, 1), 2) : 0));
    $rgb['b'] = hexdec($length == 6 ? substr($hex, 4, 2) : ($length == 3 ? str_repeat(substr($hex, 2, 1), 2) : 0));
    
    return implode( ', ', $rgb );
 }


/**
 * Check if a Raptor feature is supported/enabled.
 * 
 * @param string $feature
 * @return bool
 */
function raptor_supports( string $feature ) {
    /**
     * Filter which Raptor features can be controlled.
     * 
     * @param array The features available.
     */
    $raptor_features = apply_filters(
        'raptor_features',
        [
            'acf' => true,
            'acf/flexi' => true,
            'acf/flexi/templates' => true,
            'acf/flexi/global_blocks' => true,
            'forms' => true,
            'headless' => true,
            'selective_image_crop' => true
        ]
    );

    if ( !isset( $raptor_features[ $feature ] ) ) {
        return false;
    }

    return $raptor_features[ $feature ];
}


/**
 * Checks if active theme is Shiftr
 * 
 * @return bool
 */
function is_shiftr_active() {
    $active_theme_slug = get_option( 'stylesheet' );

    return $active_theme_slug === 'shiftr';
}


/**
 * Checks if active theme is Flex
 * 
 * @return bool
 */
function is_flex_active() {
    $active_theme_slug = get_option( 'stylesheet' );

    return $active_theme_slug === 'flex';
}


function raptor_get_flexi_field_key() {
    return apply_filters( 'raptor_flexi_field_key', 'flexi_blocks_builder-main' );
}


/**
 * Allow third party to modify field selector.
 * 
 * @param string $name
 * @return string
 */
function raptor_acf_get_field_selector( string $name ) {
    return apply_filters( 'raptor/acf/get_field_selector', $name );
}

/**
 * Add a new Fleximod.
 * 
 * @param string $name
 * @param array|null $objects
 * @param callable $callback
 * @return Fleximod
 */
function raptor_add_fleximod( string $name, $objects, $callback ) {
    /**
     * Allow developers to extend Fleximod with custom methods etc...
     * 
     * @param string $class The class to create a Fleximod with.
     * @param string $name The Fleximod name
     */
    $class = apply_filters( 'raptor/fleximod/class', 'Raptor\Fleximod', $name );

    $mod = new $class( $name, $objects, $callback );

    return $mod;
}


/**
 * Call a Raptor Mod
 * 
 * @param string $command Command Name
 * @param callable $callable Callback Function
 * @return Mod
 */
function raptor_mod( string $command, callable $callable ) {
    return new \Raptor\Mods\Mod( $command, $callable );
}


/**
 * Get the ID of the placeholder image.
 * 
 * @return int The attachment ID for the placeholder image
 */
function raptor_get_placeholder_id() {
    /**
     * Can't afford to wait for ACF to init correctly so directly access the field value.
     */
    return (int) get_option( 'options_misc_example_placeholder_image' );
}


/**
 * Get placeholder images to use in Flexi block previews
 * 
 * @param int $index
 * @param int|null $length
 * @return int|array
 */
function raptor_placeholder_image_ids( int $index = 0, $length = 1 ) {
    $placeholder_images = get_option( 'options_block_previews_placeholder_images' ) ?: [];

    /**
     * Can't use get_field() as ACF hasn't fully initiated.
     * 
     * @param array $placeholder_images
     */
    $image_ids = apply_filters( 'raptor_placeholder_image_ids', $placeholder_images );

    if ( !$image_ids ) {
        /**
         * Always return the expected type based on `$length`
         */
        return $length === 1 ? 0 : [];
    }

    $image_ids = array_map( function( $id ) {
        return absint( $id );
    }, $image_ids );

    /**
     * If length is 1, return int.
     */
    if ( $length === 1 ) {
        /**
         * As any index can be requested, we must check
         * if is set and default back to `0` is not.
         */
        if ( !isset( $image_ids[ $index ] ) ) {
            $index = 0;
        }

        return $image_ids[ $index ];
    }

    return array_slice( $image_ids, $index, $length );
}


/**
 * Get the URL for the Raptor API.
 * 
 * @since 1.3
 * @param string $pathname
 * @return string
 */
function raptor_get_api_url( string $pathname = '' ) {
    $origin = 'https://raptor-dashboard.v1.bc-staging.com';

    if ( defined( 'RAPTOR_API_DEV' ) ) {
        $origin = 'http://localhost:3010';
    }

    if ( defined( 'RAPTOR_API_URL' ) ) {
        $origin = RAPTOR_API_URL;
    }

    /**
     * Filter the URL origin for the Raptor API.
     * 
     * @param string $origin
     */
    $origin = apply_filters( 'raptor_api_url_origin', $origin );

    return trailingslashit( $origin ) . 'api/' . ltrim( $pathname, '/' );
}


/**
 * Compare a version Toolkit to Frontend. Useful to check if certain
 * features are available.
 * 
 * Examples:
 * If Frontend is the same version as Toolkit: raptor_frontend_version_compare( null, '==' )
 * If Frontend is the same or a newer version than Toolkit: raptor_frontend_version_compare( null, '>=' )
 * If Frontend is an older version than Toolkit: raptor_frontend_version_compare( null, '<' )
 * 
 * @param null|string $toolkit_version Defaults to the current version of Toolkit
 * @param null|string $operator The possible operators are: <, lt, <=, le, >, gt, >=, ge, ==, =, eq, !=, <>, ne
 * @return bool
 */
function raptor_frontend_version_compare( $toolkit_version = null, $operator = null ) {
    $theme = wp_get_theme();

    if ( !$toolkit_version ) {
        $toolkit_version = raptor()->version;
    }

    return version_compare( $theme->get( 'Version' ), $toolkit_version, $operator );
}


/**
 * Get the values for all settings in a settings group.
 * 
 * @param string $group
 * @return array
 */
function raptor_get_settings( string $group, string $tab = '', bool $format = true ): array {
    global $raptor_groups_library, $raptor_acf_setting_groups;
    /**
     * Support site options (legacy from 1.4.0)
     */
    if ( isset( $raptor_groups_library['site_options'] ) && $group === 'raptor' ) {
        if ( $tab ) {
            return get_field( raptor_acf_get_field_selector( $tab ), 'option', $format ) ?: [];
        }

        return [
            'contact_details' => get_field( raptor_acf_get_field_selector( 'contact_details' ), 'option', $format ),
            'social_media' => get_field( raptor_acf_get_field_selector( 'social_media' ), 'option', $format ),
            'branding' => get_field( raptor_acf_get_field_selector( 'branding' ), 'option', $format ),
            'tracking' => get_field( raptor_acf_get_field_selector( 'tracking' ), 'option', $format ),
            'cookies' => get_field( raptor_acf_get_field_selector( 'cookies' ), 'option', $format ),
            'announcement_bar' => get_field( 'announcement_bar', 'option', $format )
        ];
    }

    if ( $tab && isset( $raptor_acf_setting_groups[ $group ][ $tab ] ) ) {
        return get_field( raptor_acf_get_field_selector( $tab ), 'options', $format ) ?: [];
    }

    if ( !isset( $raptor_groups_library[ "raptor_settings_$group" ] ) ) {
        return [];
    }

    $acf_group = $raptor_groups_library[ "raptor_settings_$group" ];

    $fields = array_filter( $acf_group->acf_group_data['fields'], function( array $field ) {
        return $field['type'] !== 'tab';
    });

    $values = [];

    foreach ( $fields as $field ) {
        $values[ $field['name'] ] = get_field( $field['key'], 'options', $format );
    }

    return $values;
}


/**
 * Check Google reCAPTCHA is enabled.
 * 
 * @return bool
 */
function raptor_recaptcha_enabled() {
    $recaptcha_settings = raptor_get_settings( 'forms', 'recaptcha' );
    $enabled = false;

    if ( ( isset( $recaptcha_settings['site_key'] ) && !empty( $recaptcha_settings['site_key'] ) ) &&
        ( isset( $recaptcha_settings['secret_key'] ) && !empty( $recaptcha_settings['secret_key'] ) )
    ) {
        $enabled = true;
    }

    return apply_filters( 'raptor/forms/recaptcha_enabled', $enabled );
}


/**
 * Add a GraphQL query to the persisted queries list.
 */
function raptor_add_graphql_persisted_query( string $query ) {
    raptor()->headless->graphql_persisted_queries->add_query( $query );
}
