<?php
use Raptor\Forms\Form;
/**
 * Raptor Template Functions
 */

defined( 'ABSPATH' ) || exit;

/**  
 *  Raptor's own body_class function, more lightweight than core WP
 *
 *  @since 1.0
 */
function raptor_body_class( bool $use_raptor = true ) {
    global $wp_query, $post;

    /**
     * To prevent conflict issues, it's mcuh safer to use the default
     * body_class() function when using WooCommerce.
     */
    if ( !$use_raptor || function_exists( 'is_woocommerce' ) ) {
        body_class();

        return true;
    }

    $classes = [];

    if ( is_front_page() ) {
        $classes[] = 'front-page';
    }

    if ( is_home() ) {
        $classes[] = 'home';
        $classes[] = 'archive';
        $classes[] = 'archive-post';
    }

    if ( is_post_type_archive() ) {
        $classes[] = 'archive';
        $classes[] = 'archive-' . get_query_var( 'post_type' );
    }

    if ( is_category() ) {
        $category = get_queried_object();

        $classes[] = 'archive';
        $classes[] = 'category';
        $classes[] = 'cat-' . $category->slug;
        $classes[] = 'catid-' . $category->term_id;
    }

    if ( is_tax() ) {
        $term = get_queried_object();

        $classes[] = 'taxonomy';
        $classes[] = 'term-' . $term->slug;
        $classes[] = 'termid-' . $term->term_id;
    }

    if ( is_date() ) {
        $classes[] = 'date';
    }

    if ( is_404() ) {
        $classes[] = 'error404';
    }

    if ( is_search() ) {
        $classes[] = 'search';
    }

    if ( is_singular() ) {
        $classes[] = 'single-' . get_post_type( $post->ID );

        $classes[] = get_post_type( $post->ID ) . '_id-' . $post->ID;

        $template = get_post_meta( $post->ID, '_wp_page_template', true );
    
        if ( $template != '' ) {
            $template = preg_replace( '/^.*\/(.*)\.php$/', '\1', $template );

            $classes[] = 'template-' . $template;
        }
    }

    if ( is_user_logged_in() ) {
        $classes[] = 'logged-in';
    }

    /**
     * Filter the array of class names passed into the <body>
     * 
     * @param array $classes
     */
    $classes = apply_filters( 'raptor_body_class', $classes );

    echo 'class="' . join( ' ', $classes ) . '"';
}


/**  
 *  Keeping the header.php file cleaner
 *
 *  @since 1.0
 *  @return mixed|bool
 */
function raptor_head_open() {
    if ( raptor()->tracking['enabled'] && !empty( raptor()->tracking['head'] ) ) {
        echo raptor()->tracking['head'];
    }
}
add_action( 'wp_head', 'raptor_head_open', 20 );


/**  
 *  Keeping the header.php file cleaner
 *  Update from WP 5.2 - now hooked on wp_body_open()
 *
 *  @since 1.0
 *  @return mixed|bool
 */
function raptor_body_open() {
    if ( raptor()->tracking['enabled'] && !empty( raptor()->tracking['body_open'] ) ) {
        echo raptor()->tracking['body_open'];
    }
}
add_action( 'wp_body_open', 'raptor_body_open' );


/**  
 *  Keeping the footer.php file clean.
 *
 *  @since 1.0
 *  @return mixed|bool
 */
function raptor_body_close() {
    if ( raptor()->tracking['enabled'] && !empty( raptor()->tracking['body_close'] ) ) {
        echo raptor()->tracking['body_close'];
    }
}
add_action( 'wp_footer', 'raptor_body_close', 100 );


/**  
 *  Loop through an array of attributes and output string
 *
 *  @since 1.0
 *  @param array $raw The array of attributes
 *  @param bool $force_empty_values Whether an attribute with no value should be included in output
 *  @return string The attributes to be included inside HTML tag
 */
function raptor_output_attr( array $raw = [], bool $force_empty_values = false ) {
    $the_attributes = [];

    foreach ( $raw as $_key => $_value ) {

        if ( !$force_empty_values ) {
            if ( $_value == '' ) continue;
        }

        $key = strtolower( $_key );

        $attr_url = [ 'href', 'src', 'srcset' ];

        $value = in_array( $key, $attr_url ) ? esc_url( $_value ) : esc_attr( $_value );

        $the_attributes[] = $key . '="' . $value . '"';
    }

    return join( ' ', $the_attributes );    
}


/**  
 *  Return the HTML markup of an image from attachment ID or ACF field name
 *
 *  @since 1.0.5
 *  @param string|integer $id The attachment ID or ACF field name
 *  @param string $size The size of the image to output
 *  @param array $attr Attributes that should be added to the img tag
 * @return string
 */
function get_raptor_image( $id = 0, string $size = 'large', array $attr = [] ) {
    $html = '';

    // Check if string value may acutally be a integar before proceeding
    $id = intval( $id ) ? intval( $id ) : $id;

    // Capture ACF values automatically if field name is passed as $id
    if ( $id == 0 || is_string( $id ) ) {

        $acf_field = is_string( $id ) ? $id : 'image';

        if ( get_field( $acf_field ) ) {
            $id = get_field( $acf_field );

        } elseif ( get_sub_field( $acf_field ) ) {
            $id = get_sub_field( $acf_field );

        } else {
            return false;
        }
    }

    if ( $id > 0 ) {
        $html = wp_get_attachment_image( $id, $size, false, $attr );
    }

    return $html;
}


/**  
 *  Output the returned value from get_raptor_image()
 *
 *  @since 1.0
 *  @param int|string $id The attachment ID or ACF field name
 *  @param string $size The size of the image to output
 *  @param array $attr Attributes that should be added to the img tag
 */
function raptor_image( $id = 0, string $size = 'large', array $attr = [] ) {
    echo get_raptor_image( $id, $size, $attr );
}


/**  
 *  Output the html of the current post featured image
 *
 *  @since 1.0
 *  @param int $id The post ID
 *  @param string $size The size of the image to output
 *  @param array $attr Attributes that should be added to the img tag
 */
function raptor_featured_image( int $id = 0, string $size = 'large', array $attr = [] ) {
    global $post;

    if ( $id === 0 ) {
        $id = $post->ID;
    }

    // Allows ability to set placeholder images based on post type via $post
    $thumbnail_id = apply_filters( 'raptor_featured_image_thumbnail_id', get_post_thumbnail_id( $id ), $post );

    echo get_raptor_image( $thumbnail_id, $size, $attr );
}


/**
 * Get a icon.
 * 
 * @param string $icon_name
 * @param bool $raw SHould return the SVG only or in <div> wrapper
 * @return string|false The icon SVG markup, or false if file not found
 */
function get_raptor_icon( string $icon_name, bool $raw = false ) {
    $svg = '';
    $icon_path = get_template_directory() . '/assets/icon-set/' . trim( $icon_name ) . '.svg';

    /**
     * Allow the icon path to be changed for custom icons.
     * 
     * @param string $icon_path
     * @param string $icon_name
     */
    $icon_path = apply_filters( 'raptor_icon_path', $icon_path, $icon_name );
    
    if ( file_exists( $icon_path ) || preg_match( '/^https?:/', $icon_path ) ) {
        $file_contents = file_get_contents( $icon_path );

        if ( $file_contents === false ) {
            return false;
        }

        $svg = wp_kses(
            $file_contents,
            /**
             * Extend the default allowed SVG tags.
             * 
             * @param array $allowed_html
             * @param string $icon_name
             */
            apply_filters(
                'raptor_icon_wp_kses_allowed_html',
                [
                    'svg' => [
                        'width' => true,
                        'height' => true,
                        'viewbox' => true,
                        'xmlns' => true
                    ],
                    'g' => [
                        'fill' => true,
                        'transform' => true
                    ],
                    'path' => [
                        'd' => true,
                        'fill' => true,
                        'transform' => true
                    ]
                ],
                $icon_name
            )
        );

        if ( apply_filters( 'raptor_icon_raw', $raw, $icon_name ) ) {
            return $svg;
        }

        return sprintf( '<div class="raptor-icon %s">%s</div>', $icon_name, $svg );
    }

    return false;
}


/**
 * Get a icon.
 * 
 * @param string $icon_name
 * @param bool $raw SHould return the SVG only or in <div> wrapper
 * @return string|false The icon SVG markup, or false if file not found
 */
function raptor_icon( string $icon_name, bool $raw = false ) {
    echo get_raptor_icon( $icon_name, $raw );
}


/**
 * Display the archive title, supports Raptor archive pages.
 * 
 * @since 1.6.1
 * @param bool $display Echo or return title
 */
function raptor_archive_title( bool $display = true ) {
    $title = '';

    if ( is_home() ) {
        $title = single_post_title( '', false );
    } else if ( is_post_type_archive() ) {
        $post_type = get_query_var( 'post_type' );

        $archive_page_id = raptor_get_page_id( $post_type );

        if ( $archive_page_id ) {
            $archive_page = get_post( $archive_page_id );
    
            if ( $archive_page ) {
                $title = $archive_page->post_title;
            }
        } else {
            $title = get_the_archive_title();
        }

    } else {
        $title = get_the_archive_title();
    }

    if ( $display ) {
        echo $title;

    } else {
        return $title;
    }
}


/**
 * Displays the description for an archive page or taxonomy.
 * 
 * @since 1.6.1
 * @param bool $display Echo or return the archive description
 */
function raptor_archive_description( bool $display = true ) {
    $description = '';

    if ( is_home() ) {
        $posts_page_id = get_option( 'page_for_posts' );
        $posts_page = get_post( $posts_page_id );

        if ( $posts_page ) {
            $description = $posts_page->post_content;
        }

    } else if ( is_post_type_archive() ) {
        $post_type = get_query_var( 'post_type' );

        $archive_page_id = raptor_get_page_id( $post_type );

        if ( $archive_page_id ) {
            $archive_page = get_post( $archive_page_id );
    
            if ( $archive_page ) {
                $description = $archive_page->post_content;
            }
        }

    } else if ( ( is_category() || is_tax() ) && 0 === absint( get_query_var( 'paged' ) ) ) {
        $term = get_queried_object();

        if ( $term && !empty( $term->description ) ) {
            $description = $term->description;
        }
    }

    if ( empty( $description ) ) {
        return;
    }


    if ( $display ) {
        echo apply_filters( 'the_content', $description );
    } else {
        return apply_filters( 'the_content', $description );
    }
}


/**
 * Displays the featured image of the associated page for the custom post type
 * 
 * @since 1.6.3
 * @param string $size
 * @param array $attr
 */
function raptor_get_archive_thumbnail_id(  ) {
    $page_id = 0;

    if ( is_home() ) {

        $page_id = get_option( 'page_for_posts' );

    } else if ( is_post_type_archive() ) {

        $page_id = raptor_get_page_id( get_query_var( 'post_type' ) );
    }

    return get_post_thumbnail_id( $page_id );
}

/**
 * Displays the featured image of the associated page for the custom post type
 * 
 * @since 1.6.3
 * @param string $size
 * @param array $attr
 */
function raptor_archive_thumbnail( string $size = 'full', array $attr = [] ) {

    if ( is_home() ) {
        raptor_featured_image( get_option( 'page_for_posts' ), $size, $attr );

    } else if ( is_post_type_archive() ) {
        raptor_featured_image( raptor_get_page_id( get_query_var( 'post_type' ) ), $size, $attr );
    }
}


/**  
 *  Include a template part
 *
 *  @since 1.0.5
 *  @param string $template_name The template file name and path relative from theme_root/parts/
 *  @param array $args Any variables to pass into the template file
 *  @param string $template_path The directory to locate the template, relative from the theme root
 *  @return null on failure to find file
 */
function raptor_get_template( string $template_name, array $args = [], string $template_path = 'parts/' ) {
    /**
     * This allows flexibility to omit the .php extension when calling a template file
     */
    if ( !stripos( $template_name, '.php' ) ) {
        $template_name .= '.php';
    }

    $template = locate_template( $template_path . $template_name );

    if ( $template_path === 'blocks/' && defined( 'RAPTOR_MODE' ) && RAPTOR_MODE === 'development' ) {
        $block_name = str_replace( '.php', '', $template_name );

        $template = __DIR__ . "/../../../../raptor-library/flexi-block/blocks/$block_name/template.php";
    }

    if ( !file_exists( $template ) ) {
        return;
    }

    // Pass arguments into template file
    if ( !empty( $args ) && is_array( $args ) ) {

        extract( $args, EXTR_SKIP );
    }

    include $template;
}


/**  
 *  Include a template part, specific to blocks used in a ACF flexible content template
 *
 *  @since 1.0.5
 *  @param string $template_name The template file name and path relative from theme_root/blocks/
 *  @param array $args Any variables to pass into the template file
 *  @param string $template_path The directory to locate the template, relative from the theme root
 */
function raptor_get_block( string $block_name, array $args = [], string $template_path = 'blocks/' ) {
    raptor_get_template( apply_filters( 'raptor_get_block_name', $block_name ), $args, $template_path );
}


/**
 * Get the page id for a custom post type archive.
 * 
 * Supported:
 * - Custom post types registered via Raptor.
 * 
 * @since 1.6.1
 * @param string $page The page slug.
 * @return int
 */
function raptor_get_page_id( string $page ) {
    $page_id = 0;

    if ( !function_exists( 'get_field' ) ) {
        return $page_id;
    }

    $page_setup = get_field( raptor_acf_get_field_selector( 'page_setup' ), 'option' );

    if ( is_array( $page_setup ) && isset( $page_setup['raptor_' . $page . '_page_id'] ) ) {
        $page_id = $page_setup['raptor_' . $page . '_page_id'];
    }
    /**
     * Filter the page ID returned.
     * 
     * @param int $page_id
     * @param string $page
     */
    return apply_filters( 'raptor_get_page_id', $page_id ? absint( $page_id ) : -1 , $page );
}


/**
 * Dynamically get the page id for a custom post type archive.
 * 
 * @since 1.6.1
 * 
 * @param string $post_type Defaults to current post type.
 * @return int
 */
function raptor_get_page_id_for_archive( string $post_type = '' ) {
    
    if ( empty( $post_type ) ) {
        $post_type = get_query_var( 'post_type' );
    }

    return raptor_get_page_id( $post_type );
}


/**
 * Get a list of posts.
 * 
 * @return WP_Query
 */
function raptor_get_related_posts( array $args = [] ) {
    $default_args = [
        'posts_per_page' => 3,
        'post__not_in' => [ get_the_ID() ]
    ];

    if ( $post_ids = get_field( 'related_posts' ) ) {
        $default_args['post__in'] = $post_ids;
    }

    return new WP_Query(
        wp_parse_args(
            $args,
            $default_args
        )
    );
}


/**
 * Define the CSS color variables.
 */
function raptor_global_colors() {
    $branding = raptor_get_settings( 'raptor', 'branding' );

    if ( !$branding ) {
        return;
    }

    ?>
    <style id="raptor-global-colors">
        :root {
            <?php
            foreach ( $branding as $color => $value ) {
                $color = str_replace( 'color_', '', $color );
                $color = str_replace( '_', '-', $color );

                printf( '--c-%s: %s;', $color, $value );
                printf( '--c-%s-rgb: %s;', $color, raptor_hex_to_rgb( $value ) );
            }
            ?>
        }
    </style>
    <?php
}
add_action( 'wp_head', 'raptor_global_colors' );


function raptor_theme_color() {
    $branding = raptor_get_settings( 'raptor', 'branding' );

    if ( !$branding ) {
        return;
    }

    printf( '<meta name="theme-color" content="%s">', apply_filters( 'raptor_theme_color', $branding['color_primary'], $branding ) );
}
add_action( 'wp_head', 'raptor_theme_color', 50 );


/**
 * Correctly set post type archive page as current.
 *
 * @param array $menu_items Menu items.
 * @return array
 */
function raptor_nav_menu_item_current( $menu_items ) {
    $archive_page = is_archive() ? raptor_get_page_id( get_query_var( 'post_type' ) ) : 0;

    if ( !empty( $menu_items ) && is_array( $menu_items ) ) {
        foreach ( $menu_items as $key => $menu_item ) {
            $menu_id = (int) $menu_item->object_id;

            if ( $archive_page === $menu_id && $menu_item->object === 'page' ) {
                $menu_items[ $key ]->current = true;
            }

            $classes = array_filter(
                (array) get_post_meta( $menu_item->ID, '_menu_item_classes', true ),
                function( $class_name ) {
                    return $class_name ?: false;
                }
            );

            if ( array_search( 'menu-item-has-children', $menu_item->classes ) !== false ) {
                array_unshift( $classes, 'has-children' );
            }

            array_unshift( $classes, 'menu-item-id-' . $menu_item->ID );
            array_unshift( $classes, 'menu-item' );

            if ( $menu_item->current ) {
                $classes[] = 'current';
            } else if ( $menu_item->current_item_parent ) {
                $classes[] = 'current-parent';
            } else if ( $menu_item->current_item_ancestor ) {
                $classes[] = 'current-ancestor';
            }

            $menu_item->classes = $classes;
        }
    }

    return $menu_items;
}
add_filter( 'wp_nav_menu_objects', 'raptor_nav_menu_item_current', 3 );


/**
 * Hero component.
 * 
 * @param string $type Which hero should be loaded. Default `standard`
 * @param array $args Any variables to pass into the template file
 */
function raptor_hero( string $type = 'standard', array $args = [] ) {
    $defaults = [
        'attachment_id' => 0,
        'title' => '',
        'content' => ''
    ];

    $args = wp_parse_args( $args, $defaults );
    
    if ( $args['content'] && preg_match( '/<h1/', $args['content'] ) ) {
        $args['title'] = '';
    }

    /**
     * Filter the hero arguments, but before the conditional check to display the hero.
     * 
     * @param array $args
     * @param string $type
     */
    $args = apply_filters( 'raptor_hero_args', $args, $type );

    if ( empty( $args['content'] ) && empty( $args['title'] ) ) {
        return;
    }

    raptor_get_template( "hero-$type.php", $args );
}


/**  
 * Include an SVG file to inline on the page
 *
 * @param string $file The name of the file, excluding .svg extension, or full path to uploads
 * @param string|bool $dir The directory path relative to the theme root, or false if uploads path
 * @return mixed Output SVG element or bool (false) if file not found
 */
function raptor_inline_svg( string $file = '', $dir = '/assets/icons/' ) {
    // Put together file path
    $relative_path = get_template_directory() . $dir . $file . '.svg';

    if ( !$dir ) {
        $relative_path = str_replace( trailingslashit( get_bloginfo( 'url' ) ), '', $file );

    } else {
        $relative_path = get_template_directory() . $dir . $file . '.svg';
    }

    // Check if file exists
    if ( file_exists( $relative_path ) ) {
        include $relative_path;
    }

    return false;
}


/**
 * Render a Raptor Form
 * 
 * @param int|string $form_id
 */
function raptor_form( $form_id ) {
    $form = raptor_get_form_instance( $form_id );

    $form->render();
}


/**
 * Get an instance of Form
 * 
 * @param int|string $form_id
 * @return Form
 */
function raptor_get_form_instance( $form_id ) {
    if ( isset( raptor()->forms->forms[ $form_id ] ) ) {
        $form = raptor()->forms->forms[ $form_id ];
    } else {
        $form = new Form( (int) $form_id );
    }

    return $form;
}


/**
 * Register a Raptor Form via PHP.
 * 
 * @param array $args
 * @return Form
 */
function raptor_register_form( array $args ): Form {
    $defaults = [
        'title' => '',
        'name' => '',
        'id' => '',
        'settings' => [
            'recepients' => '',
            'subject' => '',
            'reply_to' => '',
            'redirect' => '',
            'data_capture' => true,
            'mail_send' => true
        ],
        'fields' => []
    ];

    if ( isset( $args['settings'] ) ) {
        $args['settings'] = wp_parse_args( $args['settings'], $defaults['settings'] );
    }

    $args = wp_parse_args( $args, $defaults );

    if ( empty( $args['id'] ) || gettype( $args['id'] ) !== 'string' ) {
        throw new Exception( 'Raptor Form ID must be set to type string' );
    }

    if ( isset( raptor()->forms->forms[ $args['id'] ] ) ) {
        throw new Exception( 'Raptor Form ID "' . $args['id'] . '" already exists' );
    }

    $form = new Form( $args );

    raptor()->forms->forms[ $args['id'] ] = $form;

    return $form;
}
