<?php
namespace Raptor;

class Headless {
    /** @var Headless\GraphQL_Persisted_Queries */
    var $graphql_persisted_queries;

    /**
     * Set hooks.
     */
    function __construct() {
        add_filter( 'preview_post_link', [ $this, 'preview_post_link' ], 10, 2 );
        add_filter( 'rest_prepare_page', [ $this, 'rest_prepare_page' ], 10, 2 );
        add_filter( 'home_url', [ $this, 'home_url' ], 10, 3 );
        add_filter( 'page_link', [ $this, 'page_link' ], 10, 1 );
        add_filter( 'post_type_link', [ $this, 'page_link' ], 10, 1 );
        add_filter( 'graphql_pre_resolve_uri', [ $this, 'graphql_pre_resolve_uri' ], 10, 2 );
        add_action( 'init', [ $this, 'remove_button_shortcode' ] );
        add_action( 'admin_notices', [ $this, 'admin_notices' ] );

        $this->graphql_persisted_queries = new Headless\GraphQL_Persisted_Queries;
    }


    static function get_frontend_url() {
        return defined( 'RAPTOR_HEADLESS_FRONTEND_URL' ) ? 
            trailingslashit( RAPTOR_HEADLESS_FRONTEND_URL ) : false;
    }


    /**
     * @return string|false
     */
    static function get_preview_secret() {
        return defined( 'RAPTOR_HEADLESS_PREVIEW_SECRET' ) ? 
            RAPTOR_HEADLESS_PREVIEW_SECRET : false;
    }

    
    /**
     * Update the preview link for a Next.js project.
     * 
     * @param string $link
     * @param WP_Post $post
     */
    function preview_post_link( string $link, \WP_Post $post ) {
        $frontend_url = self::get_frontend_url();

        if ( !$frontend_url ) {
            return $link;
        }
    
        $slug = strlen( $post->post_name ) > 0 ? $post->post_name : sanitize_title( $post->post_title );
        $post_type = get_post_type_object( $post->post_type )->graphql_single_name ?? $post->post_type;
    
        return add_query_arg(
            [
                'slug'      => $slug,
                'id'        => $post->ID,
                'post_type' => $post_type,
                'token'     => self::get_preview_secret()
            ],
            "{$frontend_url}api/preview"
        );
    }


    /**
     * Update the response data link to use the frontend URL.
     * 
     * @param WP_REST_Response $response
     * @param WP_Post $post
     */
    function rest_prepare_page( \WP_REST_Response $response, \WP_Post $post ) {
        if ( 'draft' === $post->post_status ) {

            // Manually call preview filter for draft posts.
            $response->data['link'] = get_preview_post_link( $post );
        } elseif ( 'publish' === $post->post_status ) {
            $frontend_url = self::get_frontend_url();

            // Override view link for published posts.
            if ( !$frontend_url ) {
                return $response;
            }
    
            // Handle special-case pages.
            $homepage_id   = intval( get_field( 'homepage', 'option' ) );
            $error_page_id = get_field( 'error_404_page', 'option' );
    
            if ( $post->ID === $homepage_id ) {
                // Return root FE URL for homepage.
                $response->data['link'] = $frontend_url;
            } elseif ( $post->ID === $error_page_id ) {
                // Return 404 URL for error page.
                $response->data['link'] = "{$frontend_url}404";
            } else {
                // Return URL based on post name.
                $response->data['link'] = "{$frontend_url}{$post->post_name}";
            }
        }
    
        return $response;
    }


    /**
     * Update the home URL to use the frontend URL.
     * 
     * @param string $url
     * @param string $path
     * @param string|null $schema
     */
    function home_url( string $url, string $path, $scheme = null ) {
        $frontend_url = self::get_frontend_url();

        if ( !$frontend_url ) {
            return $url;
        }
    
        // Don't redirect REST requests.
        if ( 'rest' === $scheme ) {
            return $url;
        }
    
        // Don't redirect unless in WP admin.
        if ( !is_admin() ) {
            return $url;
        }
    
        if ( !$path ) {
            return $frontend_url;
        }
    
        // Remove excess slash from beginning of path.
        $path = ltrim( $path, '/' );
    
        return "{$frontend_url}{$path}";
    }


    /**
     * Correctly manage the post permalink.
     * 
     * @param string $link
     */
    function page_link( string $link ) {
        $frontend_url = self::get_frontend_url();

        if ( !$frontend_url || is_graphql_request() ) {
            return $link;
        }
    
        return str_replace( trailingslashit( get_site_url() ), $frontend_url, $link );
    }


    /**
     * Return previous behaviour for handling the posts page.
     * 
     * @see https://github.com/wp-graphql/wp-graphql/releases
     * 
     * @param mixed $null
     * @param string $uri
     */
    function graphql_pre_resolve_uri( $null, $uri ) {
        $page_for_posts = get_option( 'page_for_posts', 0 );
    
        if ( empty( $page_for_posts ) ) {
            return $null;
        }
    
        $permalink = get_permalink( (int) $page_for_posts );
    
        if ( empty( $permalink ) || is_wp_error( $permalink ) ) {
            return $null;
        }
    
        $parsed_permalink = wp_parse_url( $permalink );
        $parsed_uri = wp_parse_url( $uri );
    
        if ( !isset( $parsed_permalink['path'] ) || !isset( $parsed_uri['path'] ) ) {
            return $null;
        }
    
        $trimmed_uri_path = rtrim( ltrim( $parsed_uri['path'], '/' ), '/' );
        $trimmed_permalink_path = rtrim( ltrim( $parsed_permalink['path'], '/' ), '/' );
    
        if ( $trimmed_permalink_path === $trimmed_uri_path ) {
            return new \WPGraphQL\Model\Post( get_post( (int) $page_for_posts ) );
        }
    
        return $null;
    }


    /**
     * Remove the shortcode from headless mode, as this is processed by Next.js server-side.
     */
    function remove_button_shortcode() {
        remove_shortcode( 'button' );
    }


    function admin_notices() {
        if ( !self::get_frontend_url() ) {
            echo '<div class="notice notice-error"><p>Raptor Headless error: frontend url is not set, consult a developer as soon as possible.</p></div>';
        }

        if ( !self::get_preview_secret() ) {
            echo '<div class="notice notice-error"><p>Raptor Headless error: preview secret is not set, consult a developer as soon as possible.</p></div>';
        }
    }
}

raptor()->headless = new Headless;
