<?php
namespace Raptor\Forms;

use Raptor\Forms\Field_Type;
/**
 * Initiates a form.
 */
class Form {
    /** @var WP_Post|null $post */
    var $post;

    /** @var int|string $id */
    var $id;

    /** @var string $render_id The rendered ID, allows the same form to appear on a page multiple times. */
    var $render_id;

    /** @var string $title */
    var $title;

    /** @var string $name */
    var $name;

    /** @var array $settings */
    var $settings;

    /** @var array $fields */
    var $fields = [];

    /** @var array $groups */
    var $groups = [];

    /**
     * Setup the class properties
     */
    function __construct( $id ) {
        $default_settings = [
            'recepients' => '',
            'subject' => '',
            'reply_to' => '',
            'redirect' => '',
            'recaptcha' => raptor_recaptcha_enabled(),
            'data_capture' => true,
            'mail_send' => true
        ];

        if ( is_int( $id ) ) {
            $post = get_post( $id );
    
            if ( $post && $post->post_type === 'raptor_form' ) {
                $this->post = $post;
                $this->id = $id;
                $this->title = $post->post_title;
                $this->name = $post->post_name;
                $this->fields = get_post_meta( $id, 'fields', true );
                $this->groups = get_post_meta( $id, 'groups', true );

                $this->settings = [
                    'recepients' => get_post_meta( $id, 'recepients', true ),
                    'subject' => get_post_meta( $id, 'subject', true ),
                    'reply_to' => get_post_meta( $id, 'reply_to', true ),
                    'redirect' => get_post_meta( $id, 'redirect', true ),
                    'recaptcha' => (bool) get_post_meta( $id, 'recaptcha', true ),
                    'mailchimp_send' => (bool) get_post_meta( $id, 'mailchimp_send', true ),
                    'mailchimp_tag' => get_post_meta( $id, 'mailchimp_tag', true ),
                    'data_capture' => (bool) get_post_meta( $id, 'data_capture', true ),
                    'mail_send' => (bool) get_post_meta( $id, 'mail_send', true )
                ];
            }

        } else if ( is_array( $id ) ) {
            $args = $id;

            $this->id = $args['id'];
            $this->title = $args['title'];
            $this->name = $args['name'];
            $this->settings = wp_parse_args( $args['settings'], $default_settings );
            $this->fields = $args['fields'];
            $this->groups = $args['groups'];
        }

        $this->fields = $this->parse_fields();
        $this->render_id = $this->id;
    }


    /**
     * Output the HTML of the form
     * 
     * @return false If fields are empty
     */
    function render() {
        global $raptor_forms_rendered;

        if ( empty( $this->fields ) ) {
            return false;
        }

        if ( isset( $raptor_forms_rendered[$this->id] ) ) {
            $this->render_id = $this->id . '-' . $raptor_forms_rendered[$this->id];
            $raptor_forms_rendered[$this->id]++;
        } else {
            $raptor_forms_rendered[$this->id] = 1;
        }

        printf( '<form method="post" class="raptor-form" id="raptor-form-%s" name="%s" data-raptor-form="%s">', $this->render_id, $this->name, $this->name );
        /**
         * Hidden fields.
         */
        printf( '<input type="hidden" name="action" value="%s" />', 'raptor_form_handler' );
        printf( '<input type="hidden" name="raptor_form_id" value="%s" />', $this->id );
        /**
         * Include the object ID of the current page.
         */
        $source = null;

        if ( is_singular() ) {
            $source = 'post-' . get_the_ID();
        } else if ( is_tax() ) {
            $source = 'term-' . get_queried_object_id();
        } else if ( is_post_type_archive() ) {
            $source = 'post-' . raptor_get_page_id_for_archive();
        }

        printf( '<input type="hidden" name="raptor_form_source" value="%s" />', $source );

        if ( !empty( $this->groups ) ) {
            echo apply_filters(
                'raptor/forms/groups/open',
                '<div class="groups">',
                $this
            );

            foreach ( $this->groups as $group ) {
                echo apply_filters(
                    'raptor/forms/group/open',
                    sprintf( '<fieldset name="%s">', $group['name'] ),
                    $group,
                    $this
                );

                echo apply_filters(
                    'raptor/forms/group/legend',
                    sprintf( '<legend>%s</legend>', $group['label'] ),
                    $group,
                    $this
                );

                do_action( 'raptor/forms/group/before', $group, $this );

                $this->render_fields( $group );

                do_action( 'raptor/forms/group/after', $group, $this );

                echo apply_filters(
                    'raptor/forms/group/close',
                    '</fieldset>',
                    $group,
                    $this
                );
            }
            
            echo apply_filters(
                'raptor/forms/groups/close',
                '</div>',
                $this
            );
        } else {
            $this->render_fields( ['name' => 'root'] );
        }

        echo '<div class="response"></div>';

        echo '</form>';
    }


    /**
     * Render the fields in a given group.
     * 
     * @param array $group
     */
    function render_fields( array $group ) {
        if ( !isset( $group['name'] ) ) {
            $group['name'] = 'root';
        }

        /**
         * Loop over fields.
         */
        echo '<div class="fields">';
        foreach ( $this->fields as $field ) {
            if ( !isset( $field['location'] ) ) {
                $field['location'] = 'root';
            }

            if ( $field['location'] !== $group['name'] ) {
                continue;
            }

            $class = $this->get_field_class( $field );

            do_action( 'raptor_forms_render_field_before', $field, $this );
            
            $class->render();

            do_action( 'raptor_forms_render_field_after', $field, $this );
        }
        echo '</div>';
    }


    /**
     * Get the class for the given field type.
     * 
     * @param array $field The field args.
     * @return Field_Type
     */
    function get_field_class( array $field ) {
        switch ( $field['type'] ) {
            case 'text':
                $class = new Field_Type\Text( $field, $this );
                break;

            case 'email':
                $class = new Field_Type\Email( $field, $this );
                break;

            case 'tel':
                $class = new Field_Type\Tel( $field, $this );
                break;

            case 'textarea':
                $class = new Field_Type\Textarea( $field, $this );
                break;

            case 'select':
                $class = new Field_Type\Select( $field, $this );
                break;

            case 'checkbox':
                $class = new Field_Type\Checkbox( $field, $this );
                break;

            case 'radio':
                $class = new Field_Type\Radio( $field, $this );
                break;

            case 'date':
                $class = new Field_Type\Date( $field, $this );
                break;

            case 'time':
                $class = new Field_Type\Time( $field, $this );
                break;

            case 'number':
                $class = new Field_Type\Number( $field, $this );
                break;

            case 'url':
                $class = new Field_Type\Url( $field, $this );
                break;

            case 'file':
                $class = new Field_Type\File( $field, $this );
                break;

            case 'submit':
                $class = new Field_Type\Submit( $field, $this );
                break;

            default:
                $class = new Field_Type\Text( $field, $this );
        }

        return apply_filters( 'raptor_forms_field_class', $class, $field, $this );
    }


    /**
     * Parse the field args.
     * 
     * @return array[]
     */
    function parse_fields() {
        $defaults = [
            'type' => 'text',
            'label' => '',
            'name' => '',
            'required' => false
        ];

        if ( empty( $this->fields ) ) {
            return $this->fields;
        }

        return array_map( function ( array $field ) use( $defaults ) {
            return wp_parse_args( $field, $defaults );
        }, $this->fields );
    }


    /**
     * Check reCAPTCHA is enabled for this form.
     * 
     * @return bool
     */
    function recaptcha_enabled() {
        /**
         * Dynamic filter to enable reCAPTCHA on a individual form.
         * 
         * @param bool $enabled
         */
        return apply_filters( "raptor/forms/recaptcha_enabled/{$this->name}", raptor_recaptcha_enabled() && $this->settings['recaptcha'] );
    }
}
