<?php
/**
 * Mailchimp API Class
 * 
 * This class is used to interact with the Mailchimp API. It is used to get the lists from Mailchimp and add them to the form settings.
 * 
 * To initialise the class and add the Mailchimp API key and region prefix to the form settings, use the following code in functions.php: add_filter( 'activate_raptor_mailchimp', __return_true );
 * 
 * @package Raptor
 */

use Raptor\ACF\Utils;
use Raptor\ACF\Field_Types;

class Raptor_Mailchimp_API {
    private $api_url;

    public function __construct() {
        $this->api_url = 'api.mailchimp.com/3.0/';
        $this->init();
    }

    /**
     * Initialize the class
     * 
     * Add the filter below to functions.php and have it set to true to enable the Mailchimp API
     * if ( class_exists( 'Raptor_Mailchimp_API' ) ) { new Raptor_Mailchimp_API(); }
     */
    public function init() {
        add_filter( 'raptor/forms/settings', [ $this, 'add_lists_to_form_settings' ], 10 );
        add_action( 'raptor_form_handler_capture_after', [$this, 'send_to_mailchimp'], 10, 3 );
    }


    /**
     * Validate the Mailchimp API fields
     */
    public function validate_api_fields( $form_settings ) {
        $api_key = $form_settings['api_keys']['mailchimp_api_key'];
        $region = $form_settings['api_keys']['mailchimp_region_prefix'];
        $list_id = $form_settings['api_keys']['mailchimp_list_id'];

        if ( empty( $api_key ) || empty( $region ) || empty( $list_id ) ) {
            return false;
        }

        return true;
    }

    /**
     * Add Mailchimp lists to the form settings array
     * 
     * @param array $form The form settings array
     */
    public function add_lists_to_form_settings( $form ) {
        $form_settings = raptor_get_settings( 'forms' );

        if ( !$this->validate_api_fields( $form_settings ) ) {
            return $form;
        }

        $api_key = $form_settings['api_keys']['mailchimp_api_key'];
        $region = $form_settings['api_keys']['mailchimp_region_prefix'];
        $list_id = $form_settings['api_keys']['mailchimp_list_id'];

        $lists = $this->get_lists( $api_key, $region, $list_id );

        if ( !empty( $lists ) ) {
            $form['mailchimp_lists'] = [
                'choices' => [],
            ];

            foreach ( $lists as $list ) {
                $form['mailchimpApiKey'] = true;
                $form['mailchimp_lists']['choices'][ $list['id'] ] = $list['name'];
            }
        }

        return $form;
    }

    /**
     * Get the lists from Mailchimp
     * 
     * @return array The lists from Mailchimp
     */
    public function get_lists( $api_key, $region, $list_id ) {
        if ( empty( $api_key ) || empty( $region ) || empty( $list_id ) ) {
            return [];
        }

        $url = sprintf( 
            'https://%s.%s/lists/%s/tag-search?count=200',
            $region, 
            $this->api_url,
            $list_id
        );

        $response = wp_remote_post( $url, [
            'headers' => [
                'Authorization' => 'apikey ' . $api_key,
            ],
        ] );

        if ( is_wp_error( $response ) ) {
            return [];
        }

        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if ( isset( $data['tags'] ) ) {
            return $data['tags'];
        }

        return [];
    }

    public function send_to_mailchimp( $form_instance, $values, $data_ID ) {

        if ( $form_instance->settings['mailchimp_send'] !== true || $values['signup'] !== 'on' ) {
            return;
        }

        $form_settings = raptor_get_settings( 'forms' );

        // if the API key is not set, return
        if ( !$this->validate_api_fields( $form_settings ) ) {
            return;
        }

        $api_key =          $form_settings['api_keys']['mailchimp_api_key'];
        $region =           $form_settings['api_keys']['mailchimp_region_prefix'];
        $list_id =          $form_settings['api_keys']['mailchimp_list_id'];
        $name =             $this->format_name( $values );
        $email =            ( isset( $values['email'] ) ) ? $values['email'] : ( ( $values['email_address'] ) ? $values['email_address'] : '' );
        $subscriber_hash =  md5( strtolower( $email ) );

        if ( empty( $email ) ) {
            return;
        }

        $url = sprintf( 
            'https://%s.%slists/%s/members',
            $region, 
            $this->api_url,
            $list_id
        );

        $fields = [
            'FNAME' => $name['first_name'],
            'LNAME' => $name['last_name'],
            'PHONE' => ( isset( $data['phone_number'] ) ) ? $values['phone_number'] : '',
        ];

        /**
         * Add fields to send to mailchimp
         * 
         * Defaults to FNAME, LNAME, PHONE and email is added separately
         */
        $fields = apply_filters( 'raptor_mailchimp_fields', $fields, $values );

        $body = json_encode(
            [
                'email_address' => $email,
                'status' => 'subscribed',
                'merge_fields' => $fields
            ]
        );
  
        $args = [
            'method' => 'POST',
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Basic ' . base64_encode( "site:{$api_key}" )
            ],
            'timeout' => 20,
            'sslverify' => true,
            'body' => $body
        ];
  
        $response = wp_remote_request( $url, $args );

        $response_body = json_decode( $response['body'] );  
        $mailchimp_response = false;
  
        if ( $response_body->title == 'Member Exists' ) {
            $mailchimp_response = 'already_subscribed';
        }
  
        if ( $response['response']['code'] == 200 && $response['response']['message'] == 'OK' ) {
            $mailchimp_response = 'subscribed';
        }

        add_post_meta( $data_ID, 'mailchimp_response', $mailchimp_response );

        $this->add_tag_to_user( $form_instance, $api_key, $region, $list_id, $subscriber_hash );
    }

    public function add_tag_to_user( $form_instance, $api_key, $region, $list_id, $subscriber_hash ) {
        $tag = $form_instance->settings['mailchimp_tag'];

        if ( empty( $tag ) ) {
            return;
        } elseif ( !$tag ) {
            return;
        }

        $url = sprintf( 
            'https://%s.%slists/%s/members/%s/tags',
            $region, 
            $this->api_url,
            $list_id,
            $subscriber_hash
        );

        $body = json_encode(
            [
                'tags' => [
                    [
                        'name' => $tag,
                        'status' => 'active'
                    ]
                ]
            ]
        );

        $args = [
            'method' => 'POST',
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Basic ' . base64_encode( "site:{$api_key}" )
            ],
            'timeout' => 20,
            'sslverify' => true,
            'body' => $body
        ];

        $response = wp_remote_request( $url, $args );
        $response_body = json_decode( $response['body'] );
    }

    public function format_name($data) {
        $name = [];
        
        if ( !array_key_exists( 'name', $data ) && !array_key_exists( 'full_name', $data ) ) {
            $name['first_name'] = ( $data['first_name'] ) ? $data['first_name'] : '';
            $name['last_name'] = ( $data['last_name'] ) ? $data['last_name'] : '';
        } else {
            if ( array_key_exists( 'name', $data ) ) {
                $fragments = explode( ' ', $data['name'] );
                $name['first_name'] = $fragments[0];
                $name['last_name'] = ( count( $fragments ) > 1 ) ? implode( ' ', array_slice( $fragments, 1 ) ) : '';
            } elseif ( array_key_exists( 'full_name', $data ) ) {
                $fragments = explode( ' ', $data['full_name'] );
                $name['first_name'] = $fragments[0];
                $name['last_name'] = ( count( $fragments ) > 1 ) ? implode( ' ', array_slice( $fragments, 1 ) ) : '';
            }
        }

        return $name;
    }
}
new Raptor_Mailchimp_API();
