<?php

namespace Raptor;

use Raptor\Headless\Nextjs;
use GraphQLRelay\Relay;

/**
 * Raptor Cache Manager.
 * 
 * Manages when cache should be purged.
 */
class Cache_Manager {
    function __construct() {
        add_action( 'cli_init', [ $this, 'cli_init' ] );
        add_action( 'wp_headers', [ $this, 'cache_control_header' ] );
        add_action( 'save_post', [ $this, 'purge_on_save_post' ], 10, 3 );
        add_action( 'acf/save_post', [ $this, 'update_global_block_store' ], 5 );
        add_action( 'wp_after_insert_post', [ $this, 'purge_posts_on_global_block_save' ], 10, 2 );
        add_filter( 'graphql_query_analyzer_graphql_keys', [ $this, 'graphql_query_analyzer_global_block_keys' ], 10, 4 );
        add_action( 'admin_bar_menu', [ $this, 'admin_bar_menu' ], 106 );
        add_filter( 'post_row_actions', [ $this, 'post_row_actions' ], 100, 2 );
        add_filter( 'page_row_actions', [ $this, 'post_row_actions' ], 100, 2 );
        add_action( 'wp_ajax_raptor_cache_purge_post', [ $this, 'process_ajax_purge_request' ] );
        add_action( 'wp_update_nav_menu', [ $this, 'purge_on_update_nav_menu' ], 10, 2 );
    }


    /**
     * Add CLI commands
     */
    function cli_init() {
        \WP_CLI::add_command( 'raptor cache purge-all', [ $this, 'cli_purge_all' ] );
        \WP_CLI::add_command( 'raptor cache purge-post', [ $this, 'cli_purge_post' ] );
        \WP_CLI::add_command( 'raptor cache purge-graphql', [ $this, 'cli_purge_graphql' ] );
        \WP_CLI::add_command( 'raptor cache setup-global-block-store', [ $this, 'setup_flexi_global_block_store' ] );
    }


    /**
     * Get post types that can be cached.
     * 
     * @return array
     */
    static function get_post_types() {
        /**
         * @deprecated
         */
        $types = apply_filters(
            'raptor_headless_nextjs_revalidate_post_types',
            [
                'post',
                'page'
            ]
        );

        /**
         * Configure which post types can be cached.
         * 
         * @param array
         */
        return apply_filters(
            'raptor/cache/post_types',
            $types
        );
    }


    /**
     * Purge a post via AJAX request.
     */
    function process_ajax_purge_request() {
        $post_id = isset( $_POST['post_id'] ) ? $_POST['post_id'] : 0;
        $success = false;

        if ( $post_id === 'all' ) {
            $success = $this->purge_all();
        } else if ( $post_id === 'global_data' ) {
            $success = Nextjs::revalidate_global_data();
        } else {
            $success = $this->purge_post( get_post( $post_id ) );
        }

        wp_send_json([
            'success' => $success
        ]);
    }


    /**
     * Purge a post cache via the WP CLI
     * 
     * @param array $args
     * @param array $assoc_args
     */
    function cli_purge_post( $args, $assoc_args ) {
        $post_id = $args[0];
        $post = get_post( $post_id );

        if ( $post ) {
            $purge = $this->purge_post( $post );
        }
    }


    /**
     * Purge all cache via the WP CLI
     * 
     * @param array $args
     * @param array $assoc_args
     */
    function cli_purge_all( $args, $assoc_args ) {
        $assoc_defaults = [
            'graphql' => false
        ];

        $assoc_args = wp_parse_args( $assoc_args, $assoc_defaults );
        $graphql = filter_var( $assoc_args['graphql'], FILTER_VALIDATE_BOOLEAN );

        /**
         * testing only - remove when going stable release
         */
        \WP_CLI::line( var_dump( $assoc_args ) );
        \WP_CLI::line( $graphql );

        $this->purge_all( $graphql );
    }


    /**
     * Purge graphql via the WP CLI
     * 
     * @param array $args
     * @param array $assoc_args
     */
    function cli_purge_graphql( $args, $assoc_args ) {
        self::graphql_purge_all();

        \WP_CLI::success( 'GraphQL queries purged.' );
    }


    /**
     * Add the Cache-Control header to HTTP requests.
     * 
     * @see https://developer.wordpress.org/reference/hooks/wp_headers/
     * 
     * @param array $headers
     */
    function cache_control_header( array $headers ): array {
        if ( $this->is_cache_available() && !is_admin() && !is_user_logged_in() ) {
            $default_max_age = WEEK_IN_SECONDS;

            if ( is_single() || is_page() ) {
                $current_date = new \DateTime();
                $mod_post_date = new \DateTime( get_the_modified_date() );

                $interval = $mod_post_date->diff( $current_date );

                /**
                 * If a post has not been modified in the last 3 months,
                 * set cache duration of 6 months
                 */
                if ( $interval->m >= 1 ) {
                    $default_max_age = MONTH_IN_SECONDS;
                }

                /**
                 * If a post has not been modified in the last 3 months,
                 * set cache duration of 6 months
                 */
                if ( $interval->m >= 3 ) {
                    $default_max_age = MONTH_IN_SECONDS * 6;
                }
            }

            $max_age = apply_filters( 'raptor/cache/cache_control_max_age', $default_max_age );

            $headers['Cache-Control'] = "max-age=$max_age, must-revalidate";
        }

        return $headers;
    }


    /**
     * Checks a cache provider is available. By default only WP Engine is supported.
     * 
     * @return bool
     */
    static function is_cache_available() {
        if ( raptor()->module_enabled( 'headless' ) ) {
            return true;
        }

        return class_exists( 'WpeCommon' ) && method_exists( 'WpeCommon', 'purge_varnish_cache' );
    }


    /**
     * Purge the cache when a post is saved
     * 
     * @param int $post_ID
     * @param \WP_Post $post
     * @param bool $update
     */
    function purge_on_save_post( int $post_ID, \WP_Post $post, bool $update ) {
        if ( !in_array( $post->post_type, self::get_post_types() ) ) {
            return;
        }
        /**
         * On Gutenberg editor, the `save_post` hook is fired twice. The first call,
         * the content hasn't acutally saved yet, so we don't want to revalidate the
         * pages just yet. Only trigger on the second call.
         */
        if ( $post->post_type == 'post' ) {
            $transient_name = 'pre_save_post_' . $post->ID;

            if ( !get_transient( $transient_name ) ) {
                set_transient( $transient_name, true, 10 );
                return;
            }
        }

        $this->purge_post( $post );

        if ( ( $posts_page_id = get_option( 'page_for_posts' ) ) && $post->post_type == 'post' ) {
            $this->purge_post( get_post( $posts_page_id ) );
        }
    }


    /**
     * On post save, check and update the global block store
     * 
     * @param int|string $post_ID
     */
    function update_global_block_store( $post_id ) {
        /**
         * string means it's an options page being saved
         */
        if ( gettype( $post_id ) === 'string' ) {
            return;
        }

        $post = get_post( $post_id );

        if ( !in_array( $post->post_type, self::get_post_types() ) && $post->post_type !== 'flexi_global' ) {
            return;
        }

        $flexi_field_key = raptor_get_flexi_field_key();
        /**
         * Populate and compare the existing values with data in $_POST
         */
        $flexi_data = get_field( $flexi_field_key, $post_id, false );
        $post_flexi_data = isset( $_POST['acf'][ $flexi_field_key ] ) ? $_POST['acf'][ $flexi_field_key ] : false;

        if ( !$post_flexi_data ) {
            return;
        }
        
        $global_block_ids = [];
        $post_global_block_ids = [];

        foreach ( $flexi_data as $row ) {
            $layout = $row['acf_fc_layout'];

            if ( isset( $row["{$layout}_settings"] ) ) {
                if ( isset( $row["{$layout}_settings"]["{$layout}_block_setting_use_global"] ) ) {
                    $global_block_ids[] = absint( $row["{$layout}_settings"]["{$layout}_block_setting_use_global"] );
                }
            }
        }

        foreach ( $post_flexi_data as $row ) {
            $layout = $row['acf_fc_layout'];

            if ( isset( $row["{$layout}_settings"] ) ) {
                if ( isset( $row["{$layout}_settings"]["{$layout}_block_setting_use_global"] ) ) {
                    $post_global_block_ids[] = absint( $row["{$layout}_settings"]["{$layout}_block_setting_use_global"] );
                }
            }
        }

        $global_block_ids = array_filter( array_unique( $global_block_ids ) );
        $post_global_block_ids = array_filter( array_unique( $post_global_block_ids ) );

        foreach ( array_unique( array_merge( $global_block_ids, $post_global_block_ids ) ) as $block_id ) {
            $store = get_post_meta( $block_id, 'usage_store', true );
            $updated_store = $store;

            if ( $store ) {
                /**
                 * Maybe remove from store
                 */
                if ( in_array( $post_id, $store ) && !in_array( $block_id, $post_global_block_ids ) ) {
                    $index = array_search( $post_id, $store );

                    unset( $store[ $index ] );
                    update_post_meta( $block_id, 'usage_store', $store );
                    continue;
                }

                if ( !in_array( $post_id, $store ) ) {
                    $updated_store[] = $post_id;
                }
            } else {
                $store = [];
                $updated_store = [ $post_id ];
            }
            /**
             * Add a post id to the store
             */
            if ( array_diff( $updated_store, $store ) ) {
                update_post_meta( $block_id, 'usage_store', $updated_store );
            }
        }
    }


    /**
     * Analyze the query and add the global Flexi block keys.]
     * 
     * @param array $arr
     * @param array $return_keys
     * @param array $skipped_keys
     * @param array $return_keys_array
     */
    function graphql_query_analyzer_global_block_keys( $arr, $return_keys, $skipped_keys, $return_keys_array ) {
        if ( !isset( $return_keys_array[3] ) ) {
            return $arr;
        }

        $page_id = $return_keys_array[3];
    
        $from_global_id = Relay::fromGlobalId( $page_id );
    
        if ( $from_global_id['type'] === 'post' ) {
            $flexi_data = get_field( raptor_get_flexi_field_key(), absint( $from_global_id['id'] ), false );
    
            $global_block_ids = [];
            
            if ( $flexi_data ) {
                foreach ( $flexi_data as $row ) {
                    $layout = $row['acf_fc_layout'];
        
                    if ( isset( $row["{$layout}_settings"] ) ) {
                        if ( isset( $row["{$layout}_settings"]["{$layout}_block_setting_use_global"] ) ) {
                            $global_block_ids[] = absint( $row["{$layout}_settings"]["{$layout}_block_setting_use_global"] );
                        }
                    }
                }
            }
    
            foreach ( $global_block_ids as $block_id ) {
                $arr['keys'] .= ' ' . Relay::toGlobalId( 'post', $block_id );
            }
        }
    
        return $arr;
    }


    /**
     * On Global Block save, purge each post it is used on.
     * 
     * @param int $post_ID
     * @param \WP_Post $post
     */
    function purge_posts_on_global_block_save( int $post_ID, \WP_Post $post ) {
        if ( $post->post_type !== 'flexi_global' ) {
            return;
        }

        if ( function_exists( 'graphql' ) && method_exists( Relay::class, 'toGlobalId' ) ) {
            $key = Relay::toGlobalId( 'post', $post->ID );

            self::graphql_purge( $key, 'post_UPDATE' );
        }

        $usage_store = get_post_meta( $post->ID, 'usage_store', true );

        if ( $usage_store ) {
            foreach ( $usage_store as $_post_id ) {
                $_post = get_post( $_post_id );

                if ( $_post ) {
                    self::purge_post( $_post );
                }
            }
        }
    }


    /**
     * Purge site when menu with assigned location is updated.
     * 
     * @param int $menu_id
     * @param array $menu_data
     */
    function purge_on_update_nav_menu( $menu_id = 0, $menu_data = [] ) {
        /**
         * Only purge if menu is assigned to a location.
         */
        if ( !isset( $_POST['menu-locations'] ) ) {
            return;
        }

        $transient_name = 'pre_save_menu_' . $menu_id;

        if ( !get_transient( $transient_name ) ) {
            set_transient( $transient_name, true, 10 );
            return;
        }

        /**
         * For Next.js headless mode, we only need to purge 
         * the global data, not _all_ data.
         */
        if ( raptor()->module_enabled( 'headless' ) ) {
            Nextjs::revalidate_global_data();
        } else {
            self::purge_all();
        }
    }


    /**
     * Purge the cache of a post
     * 
     * @return bool
     */
    static function purge_post( \WP_Post $post ) {
        if ( $post->post_status === 'draft' ) {
            return false;
        }

        if ( raptor()->module_enabled( 'headless' ) ) {
            /**
             * Must purge GraphQL query before Next.js revalidate
             */
            self::graphql_purge_post( $post->ID );
            /**
             * Next.js On-Demand Revalidation
             */
            return Nextjs::revalidate_post( $post );

        } else if ( class_exists( 'WpeCommon' ) && method_exists( 'WpeCommon', 'purge_varnish_cache' ) ) {
            /**
             * WP Engine Page Cache
             */
            $wpe = \WpeCommon::purge_varnish_cache( $post->ID );

            return $wpe;
        }

        return false;
    }


    /**
     * Purge the full site cache
     * 
     * @return bool
     */
    static function purge_all( bool $graphql_purge_all = false ) {
        /**
         * In headless mode, purge all usually refers to purging the global data.
         */
        if ( raptor()->module_enabled( 'headless' ) ) {
            /**
             * Purging all GraphQL queries isn't always required, you must opt in
             * by setting `$graphql_purge_all` to translate_user_role( $name:string, $domain:string )
             */
            if ( $graphql_purge_all ) {
                self::graphql_purge_all();
            }
            
            return Nextjs::revalidate_all_data();
        } else if ( class_exists( 'WpeCommon' ) && method_exists( 'WpeCommon', 'purge_varnish_cache' ) ) {
            /**
             * WP Engine Page Cache
             */
            return \WpeCommon::purge_varnish_cache();
        }

        return false;
    }


    /**
     * Purge the GraphQL cached query for a post
     * 
     * @param int $post_id
     * @param string $event
     */
    static function graphql_purge_post( int $post_id, string $event = 'post_UPDATE' ) {
        if ( !method_exists( Relay::class, 'toGlobalId' ) ) {
            return false;
        }
    
        $key = Relay::toGlobalId( 'post', $post_id );

        self::graphql_purge( $key, $event );
    }


    /**
     * Purge all GraphQL cached queries, use sparingly.
     * 
     * @param int $post_id
     * @param string $event
     */
    static function graphql_purge_all( string $event = 'undefined event' ) {
        self::graphql_purge( '', $event );
    }


    /**
     * Purge a GraphQL cached query
     * 
     * @param string $key
     * @param string $event
     */
    static function graphql_purge( string $key, string $event = 'undefined event' ) {
        if ( !function_exists( 'graphql_get_endpoint_url' ) ) {
            return;
        }
    
        $graphql_endpoint = preg_replace( '#^.*?://#', '', graphql_get_endpoint_url() );
    
        do_action( 'graphql_purge', $key, $event, $graphql_endpoint );
    }


    /**
     * Add the revalidate link to the admin bar.
     * 
     * @param WP_Admin_Bar $admin_bar
     */
    function admin_bar_menu( \WP_Admin_Bar $admin_bar ) {
        global $post;

        if ( !$this->is_cache_available() ) {
            return;
        }

        $admin_bar->add_menu([
            'id'    => 'raptor-cache',
            'title' => 'Cache'
        ]);

        if ( is_a( $post, 'WP_Post' ) && in_array( $post->post_type, $this->get_post_types() ) ) {
            $admin_bar->add_menu([
                'parent' => 'raptor-cache',
                'id'    => 'raptor-cache-purge-post',
                'title' => 'Clear Page',
                'href'  => sprintf( '?raptor-cache-purge=%s', $post->ID )
            ]);
        }

        if ( raptor()->module_enabled( 'headless' ) ) {
            $admin_bar->add_menu([
                'parent' => 'raptor-cache',
                'id'    => 'raptor-cache-purge-global_data',
                'title' => 'Clear Global Data',
                'href'  => '?raptor-cache-purge=global_data'
            ]);
        }

        $admin_bar->add_menu([
            'parent' => 'raptor-cache',
            'id'    => 'raptor-cache-purge-all',
            'title' => 'Clear All',
            'href'  => '?raptor-cache-purge=all'
        ]);
    }


    /**
     * Add the "Clear cache" link to the post row actions
     * 
     * @param array $actions
     * @param \WP_Post $post
     */
    function post_row_actions( array $actions, \WP_Post $post ) {
        if ( !$this->is_cache_available() ) {
            return $actions;
        }

        if ( in_array( $post->post_type, $this->get_post_types() ) && $post->post_status == 'publish' ) {
            $actions['purge_cache'] = sprintf( '<a href="?raptor-cache-purge=%s">Clear cache</a>', $post->ID );
        }
    
        return $actions;
    }


    /**
     * Create the stores for all global blocks
     */
    function setup_flexi_global_block_store() {
        $store = [];
        $post_type = self::get_post_types();
        /**
         * Flexi should not be applied to Posts
         */
        if ( $post_index = array_search( 'post', $post_type ) ) {
            unset( $post_type[ $post_index ] );
        }

        $query = new \WP_Query([
            'post_type' => $post_type,
            'posts_per_page' => 200
        ]);

        if ( $query->have_posts() ) {
            while ( $query->have_posts() ) {
                $query->the_post();

                $post_id = get_the_ID();
                $flexi = get_field( raptor_get_flexi_field_key(), $post_id, false );

                if ( $flexi ) {
                    foreach ( $flexi as $flexi_block ) {
                        $block_name = $flexi_block['acf_fc_layout'];
                        $global_block_id = 0;

                        if ( isset( $flexi_block["{$block_name}_settings"] ) ) {
                            if ( isset( $flexi_block["{$block_name}_settings"]["{$block_name}_block_setting_use_global"] ) ) {
                                $global_block_id = $flexi_block["{$block_name}_settings"]["{$block_name}_block_setting_use_global"];
                            }
                        }

                        if ( $global_block_id ) {
                            if ( isset( $store[ $global_block_id ] ) ) {
                                $store[ $global_block_id ][] = $post_id;
                            } else {
                                $store[ $global_block_id ] = [ $post_id ];
                            }
                        }
                    }
                }
            }
        }

        foreach ( $store as $id => $usage_store ) {
            update_post_meta( $id, 'usage_store', $usage_store );
        }
    }
}

new Cache_Manager();
