<?php
/**  
 *  Add Raptor login styles to the login page
 *
 *  @since 1.0
 */
function raptor_get_asset_url( string $path ) {
    if ( defined( 'RAPTOR_MODE' ) ) {
        return 'http://localhost:3000/assets/toolkit/' . $path;
    }

    return plugins_url( $path, RAPTOR_PLUGIN_FILE );
}

function raptor_login_styles() {
    wp_enqueue_style( 'raptor-login', raptor_get_asset_url( 'assets/css/login.css' ), false, raptor()->version );
}
add_action( 'login_enqueue_scripts', 'raptor_login_styles', 10 );


/**  
 * Remove unnecessary items from the admin top bar
 *
 * @since 1.0
 *
 * @global $wp_admin_bar
 */
function raptor_admin_bar_clean_up() {
    global $wp_admin_bar;

    $wp_admin_bar->remove_menu( 'customize' );
    $wp_admin_bar->remove_menu( 'new-content' );
    $wp_admin_bar->remove_menu( 'updates' );
    $wp_admin_bar->remove_menu( 'comments' );
    $wp_admin_bar->remove_menu( 'appearance' );
    $wp_admin_bar->remove_menu( 'wpengine_adminbar' );
}
add_action( 'wp_before_admin_bar_render', 'raptor_admin_bar_clean_up' );


/**  
 * Remove Posts and Comments from the admin menu if not required
 *
 * @since 1.0
 *
 * @global $raptor
 */
function raptor_admin_menu_clean_up() {
    if ( apply_filters( 'raptor_admin_hide_posts', false ) ) {
        remove_menu_page( 'edit.php' );
    }
    
    if ( apply_filters( 'raptor_admin_hide_comments', true ) ) {
        remove_menu_page( 'edit-comments.php' );
    }
}
add_action( 'admin_menu', 'raptor_admin_menu_clean_up' );


/**  
 * Add Raptor admin styles to the admin
 *
 * @since 1.0
 */
function raptor_admin_styles() {
    // Styles
    wp_enqueue_style( 'raptor-admin-styles', raptor_get_asset_url( 'assets/css/admin.css' ), false, raptor()->version );
    wp_enqueue_style( 'raptor-headless', raptor_get_asset_url( 'assets/css/headless.css' ), false, raptor()->version );

    // Used for SDK development mode only
    wp_register_script( 'raptor-sdk-runtime', 'http://localhost:3000/assets/js/runtime.js', [], null );

    if ( defined( 'RAPTOR_MODE' ) ) {
        wp_enqueue_script( 'raptor-sdk-runtime' );
    }

    // Scripts
    wp_enqueue_script( 'raptor-admin-script', raptor_get_asset_url( 'assets/js/admin.js' ), [], raptor()->version, [ 'in_footer' => true ] );
    wp_localize_script( 'raptor-admin-script', 'raptor', raptor_js_object() );
}
add_action( 'admin_enqueue_scripts', 'raptor_admin_styles', 99 );


/**
 * Enqueue scripts when logged in.
 */
function raptor_frontend_logged_in_scripts() {
    if ( is_user_logged_in() ) {
        wp_enqueue_script( 'raptor-admin-script', raptor_get_asset_url( 'assets/js/admin.js' ), [], raptor()->version, [ 'in_footer' => true ] );
        wp_localize_script( 'raptor-admin-script', 'raptor', raptor_js_object() );
    }
}
add_action( 'wp_enqueue_scripts', 'raptor_frontend_logged_in_scripts', 99 );

/**  
 * Add Raptor editor styles to the WYSIWYG editor
 *
 * @since 1.0
 */
function raptor_editor_styles() {
    $fonts = raptor()->fonts;

    if ( is_array( $fonts ) && isset( $fonts['web'] ) ) {
        foreach ( $fonts['web'] as $font ) {
            add_editor_style( $font['file'] );
        }
    }

    add_editor_style( 'assets/css/admin.css' );
}
add_action( 'admin_init', 'raptor_editor_styles' );


/**
 * Set post states for custom post types archive pages.
 * 
 * @since 1.6.1
 */
function raptor_set_display_post_states( $post_states, $post ) {
    global $raptor_post_types;

    foreach ( $raptor_post_types as $post_type => $instance ) {

        if ( raptor_get_page_id( $post_type ) === $post->ID ) {
            $post_states[ 'raptor_' . $post_type ] = $instance->plural . ' Page';
        }
    }

    return $post_states;
}
add_filter( 'display_post_states', 'raptor_set_display_post_states', 2, 10 );


/**  
 *  Removal of many unnecessary elements from the header
 *
 *  @since 1.0
 */
function raptor_head_removals() {
    remove_action( 'wp_head', 'rsd_link' );
    remove_action( 'wp_head', 'wlwmanifest_link' );
    remove_action( 'wp_head', 'wp_generator' );
    remove_action( 'wp_head', 'index_rel_link' );
    remove_action( 'wp_head', 'parent_post_rel_link', 10, 0 );
    remove_action( 'wp_head', 'start_post_rel_link', 10, 0 );
    remove_action( 'wp_head', 'adjacent_posts_rel_link_wp_head', 10, 0 );
    remove_action( 'wp_head', 'feed_links_extra', 3 );
    remove_action( 'wp_head', 'feed_links', 2 );
}
add_action( 'after_setup_theme', 'raptor_head_removals' );


/**  
 *  Remove all uses of emoji as they most likely won't be needed
 *
 *  @since 1.0
 */
function raptor_goodbye_emoji() {
    remove_action( 'admin_print_styles', 'print_emoji_styles' );
    remove_action( 'wp_head', 'print_emoji_detection_script', 7 );
    remove_action( 'admin_print_scripts', 'print_emoji_detection_script' );
    remove_action( 'wp_print_styles', 'print_emoji_styles' );
    remove_filter( 'wp_mail', 'wp_staticize_emoji_for_email' );
    remove_filter( 'the_content_feed', 'wp_staticize_emoji' );
    remove_filter( 'comment_text_rss', 'wp_staticize_emoji' );
}
add_action( 'init', 'raptor_goodbye_emoji' );


add_action( 'admin_footer', function() {
    echo '<div class="raptor raptor-overlay" aria-hidden="true"></div>';
});


/**
 * Show a notice above the custom post type archive page.
 *
 * @param WP_Post $post The current post object.
 */
function raptor_cpt_archive_page_notice( $post ) {
    global $raptor_post_types;

    $special_page_post_type = null;

    foreach ( $raptor_post_types as $post_type => $instance ) {

        if ( raptor_get_page_id( $post_type ) === $post->ID ) {
            $special_page_post_type = $instance;
        }
    }

    if ( !$special_page_post_type ) {
        return;
    }

    /**
     * Filters the message passed into the notice.
     * 
     * @param string
     * @param Raptor_Custom_Post_Type
     */
    $message = apply_filters(
        'raptor_cpt_page_notice',
        sprintf(
            'This is the %1$s page. A special archive that lists your %1$s. <a href="%2$s">Manage %1$s</a>',
            $special_page_post_type->plural,
            admin_url( 'edit.php?post_type=' . $special_page_post_type->name )
        ),
        $special_page_post_type
    );

    if ( $special_page_post_type && $message ) {
        printf( '<div class="%1$s"><p>%2$s</p></div>', esc_attr( 'notice notice-info' ), wp_kses( $message, [ 'a' => [ 'href' => true ] ] ) ); 
    }
}
add_action( 'edit_form_top', 'raptor_cpt_archive_page_notice' );


/**
 * Add the edit admin bar item to Raptor custom post type archive pages.
 * 
 * @param \WP_Admin_Bar $admin_bar
 */
function raptor_wp_admin_bar_edit_menu( \WP_Admin_Bar $wp_admin_bar ) {
    global $wp_the_query;

    if ( !is_admin() ) {
        $post_type = get_query_var( 'post_type' );
        $archive_page_id = raptor_get_page_id( $post_type );

        if ( !$archive_page_id ) {
            return;
        }

        $post_type_object = get_post_type_object( 'page' );
        $edit_post_link  = get_edit_post_link( $archive_page_id );

        if ( $post_type_object
            && is_post_type_archive( $post_type )
            && $edit_post_link
            && current_user_can( 'edit_post', $archive_page_id )
            && $post_type_object->show_in_admin_bar ) {
            $wp_admin_bar->add_node([
                'id'    => 'edit',
                'title' => $post_type_object->labels->edit_item,
                'href'  => $edit_post_link
            ]);
        }
    }    
}
add_action( 'admin_bar_menu', 'raptor_wp_admin_bar_edit_menu', 100 );
