<?php
namespace Raptor\ACF;

use WPGraphQL\ACF\Config;
use WPGraphQL\Registry\TypeRegistry;

class GraphQL_Flexi_Blocks extends Config {
    /** @var array $block_types */
    var $block_types = [];


    function init( TypeRegistry $type_registry ) {
        $this->type_registry = $type_registry;

        $this->register_unions();
        $this->register_field();
    }


    function register_unions() {
        global $raptor_blocks_library;

        $fields = acf_get_fields( 'flexi_main' );
        $flexi_field = $fields[0];
        $layouts = $flexi_field['layouts'];

        $block_types = [];

        foreach ( $raptor_blocks_library as $block ) {
            $block_name = ucfirst( self::camel_case( $block->name ) );

            $layout = isset( $layouts["layout_block__{$block->name}"] ) ? $layouts["layout_block__{$block->name}"] : null;

            register_graphql_object_type( $block_name, [
                'description' => 'The block.',
                'fields' => [
                    'blockName' => [
                        'type' => 'String', 
                        'description' => 'test',
                        'resolve' => function( $source ) use( $block_name ) {
                            return $block_name;
                        }
                    ]
                ]
            ]);

            $block_types[ $block->name ] = $block_name;
            $layout['parent'] = true;
            $this->add_field_group_fields( $layout, $block_name, true );
        }

        $this->block_types = $block_types;

        register_graphql_union_type( 'FlexiBlocks', [
            'typeNames' => $block_types,
            'resolveType' => function( $value ) use ( $block_types ) {
                return isset( $block_types[ $value['_flexi_block'] ] ) ? $block_types[ $value['_flexi_block'] ] : null;
            }
        ]);
    }


    function register_field() {
        register_graphql_field( 'RootQuery', 'flexiBlock', [
            'type' => 'FlexiBlocks',
            'description' => 'Example field added to the RootQuery Type',
            'args' => [
                'id' => [
                    'type' => [
                        'non_null' => 'ID'
                    ]
                ],
                'globalId' => [
                    'type' => 'ID'
                ]
            ],
            'resolve' => function( $root, $args, $context, $info ) {
                global $raptor_blocks_library;

                $block_name = $args['id'];

                if ( isset( $args['globalId'] ) && $args['globalId'] ) {
                    $global_id = $args['globalId'];
    
                    $block_data = Utils\get_global_block_data( $global_id, false );
    
                    if ( $block_data ) {
                        $block_data['_flexi_block'] = $block_name;

                        return $block_data;
                    }
                }

                $block = isset( $raptor_blocks_library[ $block_name ] ) ? $raptor_blocks_library[ $block_name ] : null;

                if ( !$block ) {
                    return null;
                }

                $example_data = $block->example['fields'];

                if ( isset( $block->example['settings'] ) ) {
                    $example_data['settings'] = $block->example['settings'];
                }

                if ( $block->args['block_before'] && isset( $block->example['block_before'] ) ) {
                    $example_data['block_before'] = $block->example['block_before'];
                }

                if ( $block->args['block_after'] && isset( $block->example['block_after'] ) ) {
                    $example_data['block_after'] = $block->example['block_after'];
                }

                $return_data = [ '_flexi_block' => $block_name ];

                /**
                 * Re-map the field `name` to `key`
                 */
                foreach ( $raptor_blocks_library[ $block_name ]->fields as $field ) {
                    if ( isset( $example_data[ $field['name'] ] ) ) {
                        if ( $field['type'] == 'group' ) {
                            $sub_field_data = [];

                            foreach ( $field['sub_fields'] as $sub_field ) {
                                $sub_field_data[ $sub_field['key'] ] = $example_data[ $field['name'] ][ $sub_field['name'] ];
                            }
                            $return_data[ $field['key'] ] = $sub_field_data;

                        } else if ( $field['type'] == 'repeater' ) {
                            $sub_field_data = [];

                            for ( $i = 0; $i < count( $example_data[ $field['name'] ] ); $i++ ) {
                                foreach ( $field['sub_fields'] as $sub_field ) {
                                    $sub_field_data[$i][ $sub_field['key'] ] = $example_data[ $field['name'] ][$i][ $sub_field['name'] ];
                                }
                            }
                            
                            $return_data[ $field['key'] ] = $sub_field_data;

                        } else if ( $field['type'] == 'post_object' ) {
                            if ( $field['multiple'] ) {
                                /**
                                 * Filter how many posts should be queried by default.
                                 * 
                                 * @param int $limit
                                 * @param array $field
                                 */
                                $limit = apply_filters( 'raptor_acf_post_object_field_default_limit', 3, $field );
                            } else {
                                $limit = 1;
                            }
                    
                            $post_ids = get_posts([
                                'post_type' => $field['post_type'],
                                'posts_per_page' => $limit,
                                'fields' => 'ids'
                            ]);
                    
                            $return_data[ $field['key'] ] = $post_ids;
                        } else if ( $field['type'] == 'taxonomy' ) {
                            if ( $field['multiple'] ) {
                                $terms = get_terms([
                                    'taxonomy' => $field['taxonomy'],
                                    'fields' => 'ids'
                                ]);

                                $return_data[ $field['key'] ] = $terms;
                            }
                        } else {
                            $return_data[ $field['key'] ] = $example_data[ $field['name'] ];
                        }
                    }

                    if ( $field['name'] == 'settings' ) {
                        $settings = [];

                        foreach ( $field['sub_fields'] as $sub_field ) {
                            if ( isset( $example_data['settings'][ $sub_field['name'] ] ) ) {
                                $settings[ $sub_field['key'] ] = $example_data['settings'][ $sub_field['name'] ];

                            } else {
                                $settings[ $sub_field['key'] ] = $sub_field['default_value'];
                            }
                        }

                        $return_data[ $field['key'] ] = $settings;
                    }

                    /**
                     * Handle `block_before` where field `name` is not default
                     */
                    if ( !isset( $return_data[ $field['key'] ] ) && $field['label'] == 'Block before' ) {
                        $return_data[ $field['key'] ] = $example_data['block_before'];
                    }
                    /**
                     * Handle `block_after` where field `name` is not default
                     */
                    if ( !isset( $return_data[ $field['key'] ] ) && $field['label'] == 'Block after' ) {
                        $return_data[ $field['key'] ] = $example_data['block_after'];
                    }
                }

                if ( $return_data ) {
                    return $return_data;
                }

                return null;
            }
        ]);
    }
}
