<?php
namespace Raptor\ACF;

use Raptor\API;
use Raptor\Headless;
use WP_CLI;

/**
 * Generate image previews for Flexi blocks using Raptor API.
 * 
 * Connects to the `v1/flexi-block-preview` endpoint which launches Puppeteer
 * and returns a screenshot taken with a 1280x800 viewport.
 * 
 * Current only method to generate previews is
 * using WP CLI command `raptor flexi regenerate-previews`
 */
class Flexi_Block_Preview {
    function __construct() {
        add_action( 'init', [ $this, 'rewrite_rules' ] );
        add_filter( 'query_vars', [ $this, 'query_vars' ] );
        add_filter( 'wp_robots', [ $this, 'wp_robots' ] );
        add_action( 'template_redirect', [ $this, 'render' ], 0 );
        add_action( 'template_redirect', [ $this, 'render_glossary' ], 0 );
        add_action( 'template_redirect', [ $this, 'render_style_guide' ], 0 );
        add_action( 'wp_head', [ $this, 'css' ], 0 );
        add_action( 'cli_init', [ $this, 'cli_init' ] );
        add_filter( 'raptor_js_object', [ $this, 'raptor_js_object' ] );
        add_action( 'save_post', [ $this, 'save_global_block' ], 10, 2 );
        add_filter( 'post_row_actions', [ $this, 'post_row_preview_action' ], 100, 2 );
        add_action( 'admin_bar_menu', [ $this, 'admin_bar_preview_link' ], 50 );
        add_filter( 'pre_get_document_title', [ $this, 'pre_get_document_title' ], 20 );
    }


    /**
     * Add the WP CLI command to regenerate previews.
     */
    function cli_init() {
        WP_CLI::add_command( 'raptor flexi regenerate-preview', [ __CLASS__, 'cli_regenerate_preview' ] );    
    }


    /**
     * Add the rewrite rule.
     */
    function rewrite_rules() {
        add_rewrite_rule( '^_raptor/flexi-block-preview/([a-z0-9\-]+)/?([0-9]*)/?', 'index.php?_raptor=flexi_block_preview&flexi_block_name=$matches[1]&flexi_global_id=$matches[2]', 'top' );
        add_rewrite_rule( '^_raptor/flexi-block-glossary/?', 'index.php?_raptor=flexi_block_glossary', 'top' );
        add_rewrite_rule( '^_raptor/style-guide/?', 'index.php?_raptor=dev_style_guide', 'top' );
    }


    /**
     * Register the `flexi_block_name` query var.
     * 
     * @param array $query_vars
     */
    function query_vars( array $query_vars ) {
        $query_vars[] = '_raptor';
        $query_vars[] = 'flexi_block_name';
        $query_vars[] = 'flexi_global_id';

        return $query_vars;
    }


    function wp_robots( array $robots ) {
        $_raptor = get_query_var( '_raptor' );

        if ( $_raptor ) {
            $robots['noindex'] = true;
            $robots['noarchive'] = true;
        }

        return $robots;
    }


    /**
     * Render the Flexi Block preview.
     */
    function render() {
        $flexi_block_name = get_query_var( 'flexi_block_name' );

        if ( !$flexi_block_name ) {
            return;
        }
        ?>
        <!DOCTYPE html>
        <html>
            <head>
                <meta name="viewport" content="width=1280, initial-scale=10" />
                <?php wp_head(); ?>
                <style>
                    html {
                        margin: 0 !important;
                    }
                    body {
                        min-height: 100vh;
                        display: flex;
                        align-items: center;
                    }
                    .flexi-blocks-builder {
                        width: 100%;
                    }
                    .flexi-block {
                        margin: 0 !important;
                    }
                    .flexi-block:not(.has-bg) {
                        margin: var(--s-vertical-padding) 0!important;
                    }
                </style>
            </head>
            <body class="flexi-block-preview">
                <?php
                raptor_preview_flexi_block( $flexi_block_name );
                wp_footer();
                ?>
            </body>
        </html>
        <?php

        exit;
    }


    /**
     * Render the Flexi Block glossary.
     */
    function render_glossary() {
        $_raptor = get_query_var( '_raptor' );

        if ( !$_raptor || $_raptor !== 'flexi_block_glossary' ) {
            return;
        }
        ?>
        <!DOCTYPE html>
        <html>
            <head>
                <meta charset="utf-8">
                <meta name="viewport" content="width=device-width, initial-scale=1.0">
                <?php wp_head(); ?>
            </head>
            <body class="flexi-block-glossary">
                <?php
                raptor_flexi_block_glossary();
                wp_footer();
                ?>
            </body>
        </html>
        <?php

        exit;
    }


    function render_style_guide() {
        $_raptor = get_query_var( '_raptor' );

        if ( !$_raptor || $_raptor !== 'dev_style_guide' ) {
            return;
        }

        $file = locate_template( 'templates/dev-style-guide.php' );

        if ( $file ) {
            include $file;
        }
    }


    function css() {
        if ( !get_query_var( '_raptor' ) ) {
            return;
        }
        ?>
        <style>#wpadminbar{display:none!important;}</style>
        <?php
    }


    function raptor_js_object( array $object ) {
        if ( is_admin() ) {
            global $raptor_blocks_library;

            $global_block_posts = get_posts([
                'post_type' => 'flexi_global',
                'posts_per_page' => -1
            ]);

            $global_blocks = [];
            $global_block_posts = array_map( function( \WP_Post $block ) use( &$global_blocks ) {
                $flexi_block = get_post_meta( $block->ID, 'flexi_block', true );
                $block->flexi_block = $flexi_block;

                if ( !isset( $global_blocks[$flexi_block] ) ) {
                    $global_blocks[$flexi_block] = [ $block->ID ];
                } else {
                    $global_blocks[$flexi_block][] = $block->ID;
                }

                return $block;
            }, $global_block_posts );

            $object['acf'] = [
                'flexi' => [
                    'blocks' => array_map(
                        function( Flexi_Block $block ) use( $global_blocks ) {
                            $fields = [];

                            foreach ( $block->fields as $field ) {
                                if ( $field['type'] === 'group' ) {
                                    foreach ( $field['sub_fields'] as $sub_field ) {
                                        if ( !isset( $sub_field['source'] ) ) {
                                            continue;
                                        }

                                        $fields[ $sub_field['key'] ] = $sub_field['source'];
                                    }
                                }

                                if ( $field['type'] === 'repeater' ) {
                                    foreach ( $field['sub_fields'] as $sub_field ) {
                                        if ( !isset( $sub_field['source'] ) ) {
                                            continue;
                                        }

                                        $fields[ $sub_field['key'] ] = $sub_field['source'];
                                    }
                                }

                                if ( !isset( $field['source'] ) ) {
                                    continue;
                                }

                                $fields[ $field['key'] ] = $field['source'];
                            }

                            return [
                                'name' => $block->name,
                                'label' => $block->label,
                                'description' => $block->description,
                                'previewUrl' => $block->preview_url(),
                                'fieldSources' => array_filter( $fields ),
                                'globals' => isset( $global_blocks[$block->name] ) ? $global_blocks[$block->name] : null
                            ];
                        },
                        $raptor_blocks_library
                    ),
                    'block_preview_timestamp' => get_option( 'raptor_flexi_block_preview' )
                ]
            ];
        }

        return $object;
    }


    static function cli_regenerate_preview( $args, $assoc_args ) {
        global $raptor_blocks_library;

        if ( empty( $args ) ) {
            if ( isset( $assoc_args['all'] ) ) {
                self::regenerate_all_block_previews();
            }

            if ( isset( $assoc_args['global'] ) ) {
                self::regenerate_all_global_block_previews();
            }
        } else {
            $block_name = $args[0];
            $block = isset( $raptor_blocks_library[ $block_name ] ) ? $raptor_blocks_library[ $block_name ] : null;

            $success = self::generate_preview( $block, isset( $args[1] ) ? $args[1] : 0 );

            if ( $success ) {
                WP_CLI::success( "Preview generated for $block->label block." );
            }
        }
    }


    /**
     * Loop through all Flexi blocks.
     */
    static function regenerate_all_block_previews() {
        global $raptor_blocks_library;

        if ( !empty( $raptor_blocks_library ) ) {
            $counter = array_filter( $raptor_blocks_library, [ __CLASS__, 'generate_preview' ] );
        }

        update_option( 'raptor_flexi_block_preview', time() );
        
        WP_CLI::success( count( $counter ) . ' Flexi block previews generated.' );
    }


    /**
     * Loop through all global Flexi blocks.
     */
    static function regenerate_all_global_block_previews() {
        global $raptor_blocks_library;

        $global_blocks = new \WP_Query([
            'post_type' => 'flexi_global',
            'posts_per_page' => -1
        ]);

        $log = [];

        if ( $global_blocks->have_posts() ) {
            while ( $global_blocks->have_posts() ) {
                $global_blocks->the_post();

                global $post;

                $block_name = get_post_meta( $post->ID, 'flexi_block', true );
                $block = isset( $raptor_blocks_library[ $block_name ] ) ? $raptor_blocks_library[ $block_name ] : false;
                /**
                 * @todo need to provide output to the console on success.
                 */
                $log[] = self::generate_preview( $block, $post->ID );
            }
        }

        WP_CLI::success( count( array_filter( $log ) ) . ' Flexi global block previews generated.' );
    }
    

    /**
     * Fetch the preview image from Raptor API.
     * 
     * @param Flexi_Block $block
     * @param int $global_id Used to load a global block
     * @return bool
     */
    static function generate_preview( Flexi_Block $block, int $global_id = 0 ) {
        if ( empty( $block->example ) && $global_id === 0 ) {
            if ( class_exists( 'WP_CLI' ) ) {
                WP_CLI::error( "Example data not set for $block->label block." );
            }

            return false;
        }

        $origin = home_url();
        $frontend_url = false;
        
        if ( class_exists( 'Headless' ) && Headless::is_headless() ) {
            $frontend_url = Headless::get_frontend_url();
        }

        if ( $frontend_url ) {
            $origin = $frontend_url;
        }

        if ( defined( 'RAPTOR_MODE' ) && RAPTOR_MODE === 'development' ) {
            $origin = 'http://localhost:3000';
        }

        $response = API::post(
            'v1/flexi-block-preview',
            [
                'origin' => $origin,
                'toolkit_version' => raptor()->version,
                'block' => $block->name,
                'global_id' => $global_id
            ],
            [
                'timeout' => 30
            ]
        );

        if ( is_wp_error( $response ) ) {
            if ( class_exists( 'WP_CLI' ) ) {
                WP_CLI::error( "API error for $block->label block." );
            }

            return false;
        }

        if ( is_array( $response ) && isset( $response['message'] ) ) {
            if ( class_exists( 'WP_CLI' ) ) {
                WP_CLI::error( $response['message'] );
            }
            return false;
        }

        $upload_dir = wp_upload_dir();
        $basedir = $upload_dir['basedir'];

        if ( is_multisite() ) {
            $basedir = WP_CONTENT_DIR . '/uploads';
        }
        
        $filename = "$block->name.jpeg";
        $previews_dir = $basedir . '/raptor/flexi-block-previews';

        if ( $global_id ) {
            $filename = "$block->name-$global_id.jpeg";
            $previews_dir = $basedir . '/raptor/flexi-block-previews/global';
        }

        if ( wp_mkdir_p( $previews_dir ) ) {
            $file = $previews_dir . '/' . $filename;

            file_put_contents(
                $file,
                $response
            );

            return true;
        }

        if ( class_exists( 'WP_CLI' ) ) {
            WP_CLI::error( "Couldn't create directory for $block->label block." );
        }

        return false;
    }


    /**
     * On global block save, regenerate the preview.
     * 
     * @param int $post_id
     * @param WP_Post $post
     */
    function save_global_block( int $post_id, \WP_Post $post ) {
        if ( $post->post_type !== 'flexi_global' ) {
            return;
        }

        global $raptor_blocks_library;

        $block = get_post_meta( $post_id, 'flexi_block', true );

        if ( isset( $raptor_blocks_library[ $block ] ) ) {
            self::generate_preview( $raptor_blocks_library[ $block ], $post_id );
        }
    }


    /**
     * Add preview link to post row.
     * 
     * @param array $actions
     * @param \WP_Post $post
     */
    function post_row_preview_action( array $actions, \WP_Post $post ) {
        if ( $post->post_type === 'flexi_global' && $post->post_status == 'publish' ) {
            $block = get_post_meta( $post->ID, 'flexi_block', true );

            $actions['flexi_block_preview'] = sprintf( '<a href="%s/_raptor/flexi-block-preview/%s/%d">Preview</a>', rtrim( home_url(), '/' ), $block, $post->ID );
        }
    
        return $actions;
    }


    /**
     * Add the revalidate link to the admin bar.
     * 
     * @param WP_Admin_Bar $admin_bar
     */
    function admin_bar_preview_link( \WP_Admin_Bar $admin_bar ) {
        if ( function_exists( '\get_current_screen' ) ) {
            global $post;

            if ( \get_current_screen()->id === 'flexi_global' ) {
                $block = get_post_meta( $post->ID, 'flexi_block', true );
        
                $admin_bar->add_menu([
                    'id'    => 'flexi_block_preview',
                    'title' => 'Preview Global Block',
                    'href'  => sprintf( '%s/_raptor/flexi-block-preview/%s/%d', rtrim( home_url(), '/' ), $block, $post->ID ),
                ]);

                return $admin_bar;
            }
        }
    }


    function pre_get_document_title( string $title ) {
        $flexi_block_name = get_query_var( 'flexi_block_name' );

        if ( $flexi_block_name ) {
            global $raptor_blocks_library;

            if ( isset( $raptor_blocks_library[ $flexi_block_name ] ) ) {
                $block = $raptor_blocks_library[ $flexi_block_name ];
                $title = sprintf( '%s - Flexi Block Preview', $block->label );
            }
        }

        return $title;
    }
}

new Flexi_Block_Preview;
