<?php
namespace Raptor\ACF\Utils;

use Raptor\ACF\Flexi_Builder;
use Raptor\ACF\Flexi_Block;
use Raptor\ACF\Field_Types;
use Raptor\ACF\Group;

/**
 * Returns the Raptor Blocks Library
 */
function blocks_library() {
    global $raptor_blocks_library;

    return $raptor_blocks_library;
}


/**
 * Prep the Flexi block for init.
 * 
 * @param array $args Arguments for the block
 * @param string $deprecated1 Legacy parameter for the `label`
 * @param array $fields Legacy parameter for the `fields`
 * @param array $dep_args Legacy parameter for the `args`
 */
function register_flexi_block( $args = [], $deprecated1 = '', $deprecated2 = [], $deprecated3 = [] ) {
    return new Flexi_Block( $args, $deprecated1, $deprecated2, $deprecated3 );
}


/**
 * Checks whether a block is in the Blocks Library
 * 
 * @param string $block_name The block to check my name
 * @return bool True if block exists, false if not
 */
function block_exists( string $block_name ) {
    return isset( blocks_library()[ $block_name ] );
}


/**
 * Get the ACF data for a global block builder
 * 
 * @return array
 */
function get_global_blocks() {
    global $raptor_blocks_library;

    $global_blocks = [];

    foreach ( $raptor_blocks_library as $block ) {
        $global_blocks[] = $block->name;
    }

    $builder = new Flexi_Builder(
        'global',
        $global_blocks,
        [
            'min' => 1,
            'max' => 1
        ],
        true
    );

    return $builder->get_acf_data();
}


/**
 * Get the ACF data for a Flexi Blocks Builder
 * 
 * @param string $builder_id
 * @return array
 */
function get_builder( string $builder_id ) {
    global $raptor_builder_library;

    return ( isset( $raptor_builder_library[ $builder_id ] ) ) ? $raptor_builder_library[ $builder_id ]->get_acf_data() : [];
}


/**
 * Check a global instance of a block exists.
 * 
 * @param string $block The block name.
 * @return bool
 */
function global_block_exists( string $block ) {
    $query = new \WP_Query([
        'post_type' => 'flexi_global',
        'meta_key' => 'flexi_block',
        'meta_value' => $block
    ]);

    return $query->found_posts > 0;
}


/**
 * Retrieve the data for a global block.
 * 
 * @param int $post_id The post ID for a global block.
 * @return array|bool The ACF flexible content data, or false.
 */
function get_global_block_data( int $post_id, bool $format = true ) {
    $builder = get_field( 'flexi_blocks_builder-global', $post_id, $format );

    if ( !empty( $builder ) ) {
        return $builder[0];
    }

    return false;
}


/**
 * Retrieve the block name.
 * 
 * @param int $post_id The post ID for a global block.
 * @return string The name of the block.
 */
function get_global_block_name( int $post_id ) {
    return get_post_meta( $post_id, 'flexi_block', true );
}


/**
 * Add a new setting tab to Raptor Settings
 * 
 * @param array $args
 */
function add_settings_tab( array $args ): void {
    global $raptor_acf_setting_groups;

    if ( empty( $args['label'] ) ) {
        throw new Error( 'Label must be set.' );
    }

    $defaults = [
        'group' => 'raptor',
        'label' => '',
        'fields' => [],
        'group_args' => []
    ];

    $args = wp_parse_args( $args, $defaults );
    $name = sanitize_title( str_replace( ' ', '_', $args['label'] ) );

    /**
     * Scope the field keys to the settings tab.
     */
    $args['fields'] = array_map( function( array $field ) use( $name ): array {
        $field['key'] = $name . '_' . $field['key'];

        return $field;
    }, $args['fields'] );

    $args = apply_filters( 'raptor/acf/settings/tab', $args );
    $args = apply_filters( "raptor/acf/settings/tab/$name", $args );

    if ( isset( $raptor_acf_setting_groups[ $args['group'] ] ) ) {
        $raptor_acf_setting_groups[ $args['group'] ][ $name ] = $args;
    } else {
        $raptor_acf_setting_groups[ $args['group'] ] = [
            $name => $args
        ];
    }
}


/**
 * Get the ACF fields for a Raptor Setting group.
 * 
 * @param string $group
 * @return array
 */
function register_settings_group( array $args ): void {
    global $raptor_acf_setting_groups, $raptor_groups_library;

    if ( !isset( $raptor_acf_setting_groups[ $args['name'] ] ) ) {
        return;
    }

    $default_args = [
        'name' => '',
        'title' => '',
        'menu_title' => '',
        'parent_slug' => ''
    ];

    $args = wp_parse_args( $args, $default_args );
    $group = $args['name'];

    /**
     * Dynamic hook to add setting tabs.
     */
    do_action( "raptor/acf/pre_register_settings_group/$group" );

    $fields = [];

    foreach ( $raptor_acf_setting_groups[ $group ] as $key => $setting_group ) {
        array_push(
            $fields,
            Field_Types\tab_field(
                $setting_group['label'],
                [
                    'placement' => 'left'
                ]
            ),
            Field_Types\group_field(
                $setting_group['label'],
                array_merge(
                    [
                        'sub_fields' => $setting_group['fields']
                    ],
                    $setting_group['group_args']
                )
            )
        );
    }

    $fields = apply_filters( "raptor/acf/settings/$group", $fields );


    if ( $group !== 'raptor' || ( $group === 'raptor' && !isset( $raptor_groups_library['site_options'] ) ) ) {
        new Group(
            "raptor_settings_$group",
            [
                'title' => $args['title'],
                'fields' => $fields,
                'location' => [
                    [
                        [
                            'param'     => 'options_page',
                            'operator'  => '==',
                            'value'     => "$group-settings"
                        ]
                    ]
                ]
            ]
        );
    }

    $options_page = array_merge(
        [
            'page_title'    => $args['title'],
            'menu_slug'     => "$group-settings",
            'capability'    => 'edit_posts'
        ],
        $args
    );

    unset( $options_page['name'] );
    unset( $options_page['title'] );

    acf_add_options_page( $options_page );

    /**
     * Rename 'Raptor' to 'Settings' in sub menu.
     */
    if ( $group == 'raptor' ) {
        add_submenu_page( 'raptor-settings', 'Raptor', 'Settings', 'edit_posts', 'raptor-settings' );
    }
}
