/**
 * Control the behaviour of the site header. This includes 
 * the hamburger menu and dropdown menus. It would also be where 
 * to handle a search bar should one exist.
 */
const header = document.querySelector('header.site-header')
const body = document.body

;(() => {
  const mobileMenuTrigger  = document.querySelector('#mobile-menu-trigger')
  const mobileMenuClose    = document.querySelector('#mobile-menu-close')
  const mobileMenu         = document.querySelector('.mobile-menu')
  const mobileMenuSubMenus = mobileMenu?.querySelectorAll('li.has-children') || []
  const menu               = document.querySelector('header.site-header nav.nav-primary')
  const menuSubMenus       = menu?.querySelectorAll('li.has-children') || []
  const headerWrapper      = header?.parentElement

  if (!menu) {
    return
  }
    
  /**
   * Sticky peek-a-boo header
   */
  let previousScroll = 0
  function displayStickyHeader() {
    const scrollPos = window.pageYOffset || document.documentElement.scrollTop
    const threshold = window.innerHeight / 4

    /** Sticky header can only be available after the threshold */
    if (scrollPos >= threshold) {
      header.classList.add('pre-set-sticky')

      setTimeout(() => {
        header.classList.add('set-sticky')
      }, 50)

      /** Catch scroll up */
      if (scrollPos < previousScroll) {
                
        /** Match scroll up against a threshold that must be reached */
        if (previousScroll > scrollPos + 60) {
          header.classList.add('is-visible')
          body.classList.add('header-state-active')
        } else {
          return
        }

      } else {
        header.classList.remove('is-visible')
        body.classList.remove('header-state-active')
      }

    } else if (scrollPos > previousScroll || scrollPos <= 0) {
      header.classList.remove('pre-set-sticky', 'set-sticky', 'is-visible')
      body.classList.remove('header-state-active')
    }

    previousScroll = scrollPos
  }
  window.addEventListener('scroll', displayStickyHeader, { passive: true })


  /**
   * Mobile menu
   */
  mobileMenu.addEventListener('click', e => e.stopPropagation())

  function openMobileMenu(event) {
    event.stopPropagation()

    mobileMenu.setAttribute('aria-hidden', false)
    headerWrapper.classList.add('mobile-menu-active')
    body.classList.add('no-scroll', 'overlay-active')

    setTimeout(() => {
      window.addEventListener('click', closeMobileMenu)
    }, 200)
  }
  mobileMenuTrigger.addEventListener('click', openMobileMenu)

  function closeMobileMenu(event) {
    event.preventDefault()

    mobileMenu.setAttribute('aria-hidden', true)
    headerWrapper.classList.remove('mobile-menu-active')
    body.classList.remove('no-scroll', 'overlay-active')

    setTimeout(() => {
      window.removeEventListener('click', closeMobileMenu)
    }, 200)
  }
  mobileMenuClose.addEventListener('click', closeMobileMenu)


  /**
   * Sub menu event handling
   */
  menu?.classList.add(menuSubMenus.length <= 0 ? 'has-no-sub-menus' : 'has-sub-menus')
  mobileMenu.querySelector('nav')?.classList.add(mobileMenuSubMenus.length <= 0 ? 'has-no-sub-menus' : 'has-sub-menus')
  const displayClass = 'is-visible'
  let openSubMenus = []

  // Primary sub-menus
  menuSubMenus.forEach(subMenu => {
    const { children, classList } = subMenu
    const link = children[0]
    const btn = !classList.contains('column') ? children[1] : null
    const subMenuEl = children[children.length - 1]
    let removeOpen

    if (subMenu.parentElement.parentElement.classList.contains('column')) {
      return
    }
        
    if (btn && btn?.nodeName === 'BUTTON') {
      btn.addEventListener('click', event => {
        event.preventDefault()
    
        if (classList.contains(displayClass)) {
          classList.remove(displayClass)
                    
        } else {
          classList.add(displayClass)
          openSubMenus.push(subMenu)
        }
      })
    } else {
      link.addEventListener('mouseover', event => {
        event.preventDefault()
    
        clearTimeout(removeOpen)

        if (classList.contains(displayClass) !== true) {
          classList.add(displayClass)
        }
      })
    
      link.addEventListener('mouseleave', () => {
        removeOpen = setTimeout(() => {
          classList.remove(displayClass)

          openSubMenus.forEach(_subMenu => {
            _subMenu.classList.remove(displayClass)
          })
        }, 200)
      })
    
      subMenuEl.addEventListener('mouseover', () => {
        clearTimeout(removeOpen)
      })
    
      subMenuEl.addEventListener('mouseleave', () => {
        removeOpen = setTimeout(() => {
          classList.remove(displayClass)

          openSubMenus.forEach(_subMenu => {
            _subMenu.classList.remove(displayClass)
          })
        }, 200)
      })
    }
  })

  // Mobile sub-menus
  mobileMenuSubMenus.forEach(subMenu => {
    const toggleSubMenu = (event) => {
      event.preventDefault();
      subMenu.classList.toggle(displayClass);
    }

    const parentLink = subMenu.children[0];
    const toggle = subMenu.children[1]

    /**
    * Allow parent link (with href="#") to also open the submenu
    */
    if(parentLink.href.endsWith('#')) {
      parentLink.addEventListener('click', toggleSubMenu)
    }

    toggle?.addEventListener('click', toggleSubMenu)
  })
})();
