/*global raptor, grecaptcha*/
import RaptorComponent from '../../lib/component-functions'


/**
 * class
 */
export default class Form extends RaptorComponent {

  /** @var {bool} cssRequested State of the stylesheet download for Form actions */
  cssRequested = false

  /** @var {object} storage The user input from localStorage */
  storage = {}

  componentSlug() {
    return 'form'
  }

  defaultSettings() {
    return {
      validate: true,
      submission: true,
      validationClasses: {
        focus: 'field_event--focus',
        success: 'field_event--success',
        error: 'field_event--error'
      },
      submitMessageCopy: raptor.form || {}
    }
  }


  /**
   * 
   */
  constructor(...args) {
    super(...args)

    this.fields = this.target.querySelectorAll('input:not([type="hidden"]):not([type="submit"]), textarea, select')
    this.submitButton = this.target.querySelector('[type="submit"]')
    this.submitButtonOrgLabel = this.submitButton.innerHTML

    this.actionClass = this.settings.validationClasses
    this.storage = this.getStorage()
  }


  /**
   * Set up event listeners to all components of the form
   */
  init() {
    this.loadRecaptcha()
    this.fieldEvents()
    this.submitButton.addEventListener('mouseover', this.getStyles.bind(this));
    this.target.addEventListener('submit', this.submit.bind(this))
  }

  /**
   * Load reCAPTCHA script
   * 
   * @returns {void}
   */
  loadRecaptcha() {
    if (!raptor.form.recaptchaSiteKey && !document.querySelector('#raptor-recaptcha-js')) {
      return
    }

    const script = document.createElement('script')
    script.type = 'text/javascript'
    script.src = `https://www.google.com/recaptcha/api.js?render=${raptor.form.recaptchaSiteKey}`
    script.id = 'raptor-recaptcha-js'
    document.body.appendChild(script)
  }


  /**
   * 
   */
  fieldEvents() {
    const { focus, success, error } = this.actionClass

    this.fields.forEach(field => {
      const { parentElement: fieldWrapper } = field
      field.validateErrorActive = false

      if (field.name in this.storage && this.storage[field.name]) {
        if (field.type == 'checkbox') {
          if (field.name.match(/\[\]$/)) {
            if (this.storage[field.name].indexOf(field.value) >= 0) {
              field.checked = true
            }
          } else {
            field.checked = true
          }
        } else if (field.type == 'radio') {
          if (field.value === this.storage[field.name]) {
            field.checked = true
          }
        } else {
          field.value = this.storage[field.name]
        }
      }

      if (field.nodeName === 'SELECT') {
        field.addEventListener('change', () => {
          this.updateStorage(field)
          fieldWrapper.classList.add(success)
        })

      } else if (field.type === 'file') {
        fieldWrapper.querySelector('.remove-files').addEventListener('click', event => {
          event.preventDefault()
                    
          field.value = ''
          fieldWrapper.classList.remove('with-files', success)
        })

        field.addEventListener('change', () => {
          const html = [...field.files].map(file => {
            return `<li>${file.name}</li>`
          })
                    
          fieldWrapper.querySelector('.uploaded-files').innerHTML = html.join('')
          fieldWrapper.classList.remove(focus, success, error)

          if (field.files) {
            fieldWrapper.classList.add('with-files', success)
          } else {
            fieldWrapper.classList.remove('with-files')
          }
        })
      } else {
        if (field.type != 'checkbox' && field.type != 'radio') {

          field.addEventListener('mouseover', () => {
            this.showValidationError(field)
          })

          field.addEventListener('mouseleave', () => {
            this.hideValidationError(field)
          })
                    
          field.addEventListener('focus', () => {
            this.getStyles()
            this.clearValidationError(field)
            fieldWrapper.classList.add(focus)
          })
    
          field.addEventListener('blur', () => {
            this.updateStorage(field)
            fieldWrapper.classList.remove(focus, success, error)
    
            if (this.settings.validate && field.value != '') {
    
              if (field.checkValidity()) {
                fieldWrapper.classList.add(success)
                this.clearValidationError(field)
    
              } else  {
                fieldWrapper.classList.add(error)
              }
            }
          })

        } else if (field.type == 'checkbox' || field.type == 'radio') {
          if (this.settings.validate) {
            field.addEventListener('change', () => {
              this.updateStorage(field)
        
              if (field.checked) {
                fieldWrapper.classList.remove(error)
                this.clearValidationError(field)
              }
            })
          }
        }

        if (this.settings.validate) {
          field.addEventListener('invalid', e => {
            e.preventDefault()

            fieldWrapper.classList.add(error)
            if (! field.validateErrorActive) {
              this.displayValidationError(field)
              field.validateErrorActive = true
            } else {
              this.showValidationError(field)
            }
          })
        }
      }
    })

    const checkboxMultipleFields = this.target.querySelectorAll('.field.field_type--checkbox fieldset')

    checkboxMultipleFields.forEach(fieldset => {
      if (fieldset.dataset.required) {
        this.checkboxMultiple(fieldset)
      }
    })
  }


  /**
   * Generate and display the validation notice of a field
   * 
   * @param {object} field The form field element
   */
  displayValidationError(field) {
    const { parentElement: fieldWrapper } = field
    const notice = document.createElement('span')

    notice.classList.add('validation')
    notice.innerHTML = field.validationMessage
    fieldWrapper.appendChild(notice)

    setTimeout(() => {
      notice.classList.add('is-visible')
    }, 200)

    setTimeout(() => {
      this.hideValidationError(field)
    }, 8000)
  }


  showValidationError(field) {
    const { parentElement: fieldWrapper } = field
    const notice = fieldWrapper.querySelector('span.validation')
        
    if (notice) {
      notice.classList.add('is-visible')
    }
  }


  hideValidationError(field) {
    const { parentElement: fieldWrapper } = field
    const notice = fieldWrapper.querySelector('span.validation')
        
    if (notice) {
      notice.classList.remove('is-visible')
    }
  }


  /**
   * Clear the validation notice of a field
   * 
   * @param {object} fieldWrfieldapper The form field element
   */
  clearValidationError(field) {
    const { parentElement: fieldWrapper } = field
    const notice = fieldWrapper.querySelector('span.validation')
        
    if (notice) {
      notice.classList.remove('is-visible')

      setTimeout(() => {
        fieldWrapper.removeChild(notice)
        field.validateErrorActive = false
      }, 400)
    }
  }


  /**
   * Send the data to the server for handling
   * 
   * @param {object} e The event
   */
  async submit(e) {
    e.preventDefault()

    this.target.classList.add('send-in-progress')
    this.submitButton.disabled = true
    this.submitButton.innerHTML = 'Sending...'

    let token = ''

    if (typeof grecaptcha !== 'undefined' && raptor.form.recaptchaSiteKey) {
      token = await grecaptcha.execute(raptor.form.recaptchaSiteKey, { action: 'form_submit' })
    }


    const body = new FormData(this.target)
    body.append('grecaptcha-token', token)

    fetch(raptor.ajaxUrl, {
      method: 'post',
      body
    })
      .then(res => res.json())
      .then(data => {
        this.target.classList.remove('send-in-progress')
        this.response = data

        if (data.success) {
          this.removeStorage()
        }

        if (data.success && data.redirect) {
          location.href = data.redirect
        }

        this.displaySubmitMessage(data)
      })
      .catch(error => {
        console.log(error)
      })
  }


  /**
   * @param {object} data The body of the response
   */
  displaySubmitMessage(data = {}) {
    this.target.classList.add('submitted')

    const submitResponse = this.target.querySelector('.response')
    const svgDefaultAtts = 'xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 0 24 24" width="24px"'

    submitResponse.classList.add(data.success ? 'success' : 'error')

    submitResponse.innerHTML = `
      <div>
        ${data.success ? `
          <svg ${svgDefaultAtts} fill="var(--c-primary)">
            <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm0 18c-4.41 0-8-3.59-8-8s3.59-8 8-8 8 3.59 8 8-3.59 8-8 8zm3.88-11.71L10 14.17l-1.88-1.88c-.39-.39-1.02-.39-1.41 0-.39.39-.39 1.02 0 1.41l2.59 2.59c.39.39 1.02.39 1.41 0L17.3 9.7c.39-.39.39-1.02 0-1.41-.39-.39-1.03-.39-1.42 0z"/>
          </svg>
        ` : `
          <svg ${svgDefaultAtts} fill="var(--c-red,#ff0000)">
            <path d="M12 7c.55 0 1 .45 1 1v4c0 .55-.45 1-1 1s-1-.45-1-1V8c0-.55.45-1 1-1zm-.01-5C6.47 2 2 6.48 2 12s4.47 10 9.99 10C17.52 22 22 17.52 22 12S17.52 2 11.99 2zM12 20c-4.42 0-8-3.58-8-8s3.58-8 8-8 8 3.58 8 8-3.58 8-8 8zm1-3h-2v-2h2v2z"/>
          </svg>
        `}
        <p>${data.message}</p>
        <button type="button" name="reset_form">Reset form</button>
      </div>
    `

    submitResponse.querySelector('button[name="reset_form"]').addEventListener('click', event => {
      event.preventDefault()

      this.clearSubmitMessage(data)
    })

    /**
     * Support for Google Tag Manager tracking
     */
    if ((data.success && !data.test_mode) || data.test_mode === 'test-tracking') {
      window.dataLayer = window.dataLayer || []
      window.dataLayer.push({
        'event': 'formSubmission',
        'form': this.target.dataset.raptorForm
      })
    }

    setTimeout(() => {
      submitResponse.setAttribute('aria-hidden', false)

      this.submitButton.disabled = false
      this.submitButton.innerHTML = this.submitButtonOrgLabel
    }, 100)
  }


  /**
   * @param {object} message The message wrapper element
   * @param {string} data The server response
   */
  clearSubmitMessage(data) {

    if (data.success) {
      const { success, error } = this.actionClass

      this.target.reset()

      // Empty all fields of values
      for (let i = 0; i < this.fields.length; i++) {
        const field = this.fields[i]

        if (field.nodeName === 'SELECT' || (field.type != 'checkbox' && field.type != 'radio')) {
          field.parentElement.classList.remove(success, error)
        }
      }
    }

    setTimeout(() => {
      this.target.classList.remove('submitted')
      this.target.querySelector('.response').setAttribute('aria-hidden', true)
    }, 100)
        
    /**
     * Enabling the button again is the very last thing we need to do
     */
    this.submitButton.disabled = false
    this.submitButton.innerHTML = this.submitButtonOrgLabel
  }


  /**
   * Request the stylesheet for Form interactions
   */
  getStyles() {
    let assetPath = raptor.theme

    if (! this.cssRequested) {
      const stylesheet = document.createElement('link');
      stylesheet.setAttribute('rel', 'stylesheet');
      stylesheet.setAttribute('href', `${assetPath}/assets/css/form.css`);
      stylesheet.setAttribute('type', 'text/css');
      const head = document.getElementsByTagName('head')[0]
      head.appendChild(stylesheet);

      this.cssRequested = true;
    }

    this.submitButton.removeEventListener('mouseover', this.getStyles);
  }

  /**
   * Support multiple checkbox as a required field.
   * 
   * @param {HTMLElement} fieldset 
   */
  checkboxMultiple(fieldset) {
    fieldset.addEventListener('change', () => {
      if (fieldset.querySelector('input[type="checkbox"]:checked')) {
        fieldset.querySelectorAll('input[type="checkbox"]').forEach(checkbox => {
          checkbox.removeAttribute('required')
        })
      } else {
        fieldset.querySelectorAll('input[type="checkbox"]').forEach(checkbox => {
          checkbox.setAttribute('required', '')
        })
      }
    })
  }

  /**
   * Get the key to use in localStorage.
   * 
   * @returns {string}
   */
  getStorageKey() {
    return `raptor_form_${this.target.dataset.raptorForm}`
  }

  /**
   * Get stored user input from localStorage
   * 
   * @returns {object}
   */
  getStorage() {
    const storage = localStorage.getItem(this.getStorageKey())

    if (!storage) {
      return {}
    }

    return JSON.parse(storage)
  }

  /**
   * Update the values in the storage
   * 
   * @param {HTMLElement} field The field element
   */
  updateStorage(field) {
    const storage = this.getStorage()

    if (field.type == 'checkbox') {
      if (field.name.match(/\[\]$/)) {
        if (field.checked) {
          if (field.name in storage) {
            storage[field.name].push(field.value)
          } else {
            storage[field.name] = [field.value]
          }
        } else {
          if (field.name in storage) {
            const index = storage[field.name].indexOf(field.value)
            storage[field.name].splice(index, 1)
          }
        }
      } else {
        storage[field.name] = field.checked ? field.value : ''
      }
    } else if (field.type == 'radio') {
      storage[field.name] = field.checked ? field.value : ''
    } else {
      storage[field.name] = field.value
    }

    localStorage.setItem(this.getStorageKey(), JSON.stringify(storage))
    this.storage = storage
  }


  /**
   * Remove the user input from localStorage
   */
  removeStorage() {
    localStorage.removeItem(this.getStorageKey())
  }
}
