<?php

namespace Raptor\Headless;

use \WPGraphQL\SmartCache;
use \GraphQL\Language;
use WP_CLI;

class GraphQL_Persisted_Queries {
    /** @var string[] */
    var $queries = [];

    /** @var array[] */
    var $query_hashes = [];

    function __construct() {
        add_action( 'cli_init', [ $this, 'cli_init' ] );
    }

    function cli_init() {
        WP_CLI::add_command(
            'raptor headless reset-persisted-queries',
            [ $this, 'reset_persisted_queries' ]
        );
        WP_CLI::add_command(
            'raptor headless set-graphql-cache-section',
            [ $this, 'set_graphql_cache_section' ]
        );
    }

    function set_graphql_cache_section() {
        $graphql_cache_section = get_option( 'graphql_cache_section' );

        if ( $graphql_cache_section ) {
            $graphql_cache_section['global_max_age'] = 15552000;

        } else {
            $graphql_cache_section = [
                'log_purge_events' => 'off',
                'global_max_age' => 15552000,
                'cache_toggle' => 'off',
                'global_ttl' => NULL,
                'purge_all' => false,
                'purge_all_timestamp' => false,
            ];

            add_option( 'graphql_cache_section', $graphql_cache_section );
        }
    }


    /**
     * @return string
     */
    static function get_flexi_blocks( string $post_type = 'page' ) {
        global $raptor_blocks_library;
    
        return array_reduce( $raptor_blocks_library, function( $carry, $flexi_block ) use( $post_type ) {
            $block_name = $flexi_block->args['graphql_name'];
            $gql = $flexi_block->args['graphql_query'];
            $field_group_name = $post_type === '_flexi_preview' ? '' : 'fieldGroupName';
            $__typename = $post_type === '_flexi_preview' ? '__typename' : '';

            $layout_type_name = apply_filters(
                'raptor/headless/graphql/flexi_layout_type_name',
                "FlexiBlocksBuilderFlexiBlocksBuilder{$block_name}Layout",
                $post_type,
                $block_name
            );
    
            if ( $gql ) {
                $carry .= "\n
                ... on {$layout_type_name} {
                    $__typename
                    $field_group_name
                    $gql
                }
            ";
            }
    
            return $carry;
        }, '' );
    }


    function flexi_block_preview_query() {
        $flexi_gql = $this->get_flexi_blocks( '_flexi_preview' );

        $this->add_query("
            query getFlexiBlockPreview(\$id: ID!, \$globalId: ID) {
                flexiBlock(id: \$id, globalId: \$globalId) {
                    $flexi_gql
                }
            }
        ");
    }

    function reset_persisted_queries() {
        $this->delete_queries();

        /**
         * Load files from theme directory.
         */
        $theme_dir = get_template_directory();

        foreach ( glob( $theme_dir . '/graphql/fragments/*.php' ) as $fragment_file ) {
            include_once $fragment_file;
        }

        foreach ( glob( $theme_dir . '/graphql/*.php' ) as $query_file ) {
            include_once $query_file;
        }

        $cli_output = [];

        foreach ( $this->queries as $query ) {
            $cli_output[] = $this->generate_query_hash( $query );
        }

        if ( empty( $cli_output ) ) {
            WP_CLI::error( 'No persisted queries found.' );
            return;
        }

        WP_CLI\Utils\format_items( 'table', $cli_output, [ 'query', 'hash' ] );
    }


    function repopulate_persisted_queries() {
        $this->delete_queries();

        /**
         * Load files from theme directory.
         */
        $theme_dir = get_template_directory();

        foreach ( glob( $theme_dir . '/graphql/fragments/*.php' ) as $fragment_file ) {
            include_once $fragment_file;
        }

        foreach ( glob( $theme_dir . '/graphql/*.php' ) as $query_file ) {
            include_once $query_file;
        }

        $cli_output = [];

        foreach ( $this->queries as $query ) {
            $cli_output[] = $this->generate_query_hash( $query );
        }

        if ( empty( $cli_output ) ) {
            WP_CLI::error( 'No persisted queries found.' );
            return;
        }

        // create json file with queries
        $json_file = ABSPATH . '/persisted_queries.json';
        file_put_contents( $json_file, json_encode( $cli_output ) );
    }


    function delete_queries() {
        $existing_queries = new \WP_Query([
            'post_type' => 'graphql_document',
            'posts_per_page' => -1
        ]);

        if ( $existing_queries->have_posts() ) {
            while ( $existing_queries->have_posts() ) {
                $existing_queries->the_post();

                wp_delete_post( get_the_ID() );
            }
        }
    }


    function add_query( string $query ) {
        $this->queries[] = $query;
    }


    function generate_query_hash( string $query ) {
        if ( !class_exists( '\WPGraphQL\SmartCache\Utils' ) ) {
            return false;
        }
    
        $ast = Language\Parser::parse( $query );
        $query = Language\Printer::doPrint( $ast );
        $normalized_hash = SmartCache\Utils::getHashFromFormattedString( $query );
    
        $post = SmartCache\Utils::getPostByTermName( $normalized_hash, SmartCache\Document::TYPE_NAME, SmartCache\Document::ALIAS_TAXONOMY_NAME );
    
        if ( empty( $post ) ) {
            $operation_names = [];
    
            $definition_count = $ast->definitions->count();
            for ( $i = 0; $i < $definition_count; $i++ ) {
                $node              = $ast->definitions->offsetGet( $i );
                $operation_names[] = isset( $node->name->value ) ? $node->name->value : __( 'A Persisted Query', 'wp-graphql-smart-cache' );
            }
    
            $post_id = wp_insert_post([
                'post_title' => join( ', ', $operation_names ),
                'post_content' => Language\Printer::doPrint( $ast ),
                'post_name'    => $normalized_hash,
                'post_status'  => 'publish',
                'post_type' => SmartCache\Document::TYPE_NAME
            ]);
        
            wp_add_object_terms( $post_id, [ $normalized_hash ], SmartCache\Document::ALIAS_TAXONOMY_NAME );
    
            $result = [
                'query' => array_pop( $operation_names ),
                'hash' => $normalized_hash
            ];

            $this->query_hashes[] = $result;

            return $result;
        }
    }
}
