<?php

namespace Raptor\Forms;

class Export_Submissions {
    function __construct() {
        add_action( 'admin_init', [ $this, 'admin_init' ] );
    }


    function admin_init() {
        $export = isset( $_GET['raptor-form-submissions-export'] );

        if ( !current_user_can( 'manage_options' ) ) {
            return;
        }

        if ( $export ) {
            $this->download( $this->export() );
        }
    }


    /**
     * Export form submission data categorized by forms.
     * 
     * @return array[]
     */
    function export() {
        $query_args = [
            'post_type' => 'raptor_form_data',
            'posts_per_page' => -1,
            'post_status' => 'all'
        ];

        $export_form = isset( $_GET['raptor-form-id'] ) ? absint( $_GET['raptor-form-id'] ) : 0;

        if ( $export_form !== 0 ) {
            $query_args['meta_key'] = 'form_id';
            $query_args['meta_value'] = $export_form;
        }

        $m = isset( $_GET['m'] ) ? $_GET['m'] : false;

        if ( $m ) {
            $query_args['m'] = $m;
        }

        $data = new \WP_Query( $query_args );

        $forms = [];

        if ( $data->have_posts() ) {
            while ( $data->have_posts() ) {
                $data->the_post();

                $data_id = get_the_ID();
                $form_id = get_post_meta( $data_id, 'form_id', true );

                $form = raptor_get_form_instance( $form_id );
                $submission_data = raptor_get_form_data( $data_id );

                $formatted_submission_data = array_merge(
                    [ $data_id, $form->title, get_the_date( 'd/m/y H:i' ) ],
                    array_map( function( $item ) {
                        return $this->maybe_quotes_wrap( $this->filter_content_field( $item ) );
                    }, array_values( $submission_data ) )
                );

                if ( isset( $forms[ $form_id ] ) ) {
                    $forms[ $form_id ]['submissions'][ $data_id ] = $formatted_submission_data;

                } else {
                    $forms[ $form_id ] = [
                        'fields' => array_merge(
                            [ 'id', 'form', 'date' ],
                            array_keys( $submission_data )
                        ),
                        'submissions' => [
                            $data_id => $formatted_submission_data
                        ]
                    ];
                }
            }
        }

        return $forms;
    }


    function download( array $data ) {
        $url = get_home_url();
        $parsed_url = wp_parse_url( $url );
        $hostname = preg_replace( '/^www\./', '', $parsed_url['host'] );
        $hostname = str_replace( '.', '_', $hostname );
        $file_name = $hostname . '-form-submissions-' . date( 'Y-m-d' ) . '.csv';

        header( 'Content-Type: text/csv' );
        header( "Content-Disposition: attachment; filename={$file_name}" );
        
        $fp = fopen( 'php://output', 'wb' );

        foreach ( $data as $form ) {
            fputcsv( $fp, $form['fields'] );

            foreach ( $form['submissions'] as $submission ) {
                fputcsv( $fp, $submission );
            }
        }

        fclose( $fp );
        die;
    }

    /**
     * Filter content field for export.
     * Convert newlines to '\n'.
     *
     * @param string $content
     * @return string
     */
    protected function filter_content_field( $content ) {
        $content = str_replace( '\n', "\\\\n", $content );

        return $content;
    }


    /**
     * Check if a string contains a comma and return string wrapped in double quotes.
     * 
     * @param string $string
     * @return string
     */
    function maybe_quotes_wrap( $string ) {
        $content = htmlspecialchars( $string, ENT_QUOTES );

        return $content;
    }
}

new Export_Submissions;
