<?php
/**
 * The reason the majority of these functions exist is to reduce
 * the amount of code acutally within the template file. This helps
 * reduce noise so you, the developer, can focus on what matters.
 */
use Raptor\ACF\Utils as Utils;
use Raptor\ACF\Field_Types;


/**
 * Configure the global `$flexi_blocks` that holds all block data.
 */
$GLOBALS['flexi_blocks'] = [];

/**
 * Setup the `$flexi_blocks` global variable.
 * 
 * @param null|int|WP_Post|WP_Term $object
 * @param string $builder The Flexi Blocks Builder name
 */
function raptor_flexi_blocks_setup( $object = null, string $builder = 'main' ) {
    global $flexi_blocks;

    if ( !$object ) {
        if ( is_tax() ) {
            $object = get_queried_object();
        }
    
        if ( is_archive() ) {
            if ( is_tax() && function_exists( 'taxonomy_is_product_attribute' ) ) {
                if ( !taxonomy_is_product_attribute( $object->taxonomy ) ) {
                    $object = raptor_get_page_id_for_archive();
                }
            } else {
                $object = raptor_get_page_id_for_archive();
            }
        }
        
        if ( is_home() ) {
            $object = get_post( get_option( 'page_for_posts' ) );
        }
    
        if ( function_exists( 'is_woocommerce' ) && is_shop() ) {
            $object = wc_get_page_id( 'shop' );
        }
    }

    $flexi_blocks = raptor_flexi_parse_global_blocks_data_formatted( get_field( raptor_get_flexi_field_key(), $object ) ?: [] );
}
add_action( 'template_redirect', 'raptor_flexi_blocks_setup' );


/**
 * Outputs the whole Flexi Blocks Builder HTML
 * 
 * @param string $builder_name The suffix to the full name
 * @param mixed $flexible_content_object The post/object to get data from
 */
function raptor_flexi_blocks_builder( string $builder_name = 'main', $flexible_content_object = false ) {
    raptor_flexi_blocks_setup( $flexible_content_object );

    if ( !$flexible_content_object ) {
        if ( is_singular( 'page' ) ) {
            global $post;
            $flexible_content_object = $post;
    
        } else if ( is_tax() ) {
            $flexible_content_object = get_queried_object();

        } else if ( is_post_type_archive() ) {
            global $raptor_post_types;

            $current_post_type = get_query_var( 'post_type' );
            $post_type_page_id = 0;

            if ( isset( $raptor_post_types[ $current_post_type ] ) ) {
                $post_type_page_id = raptor_get_page_id( $current_post_type );
            }

            if ( function_exists( 'is_woocommerce' ) && is_shop() ) {
                $post_type_page_id = wc_get_page_id( 'shop' );
            }

            if ( $post_type_page = get_post( $post_type_page_id ) ) {
                $flexible_content_object = $post_type_page;
            }
            
        } else if ( is_home() ) {
            $posts_page_id = get_option( 'page_for_posts' );
            $posts_page = get_post( $posts_page_id );

            if ( $posts_page ) {
                $flexible_content_object = $posts_page;
            }
        }
    }

    /**
     * Filter the object to use for the ACF field data.
     * 
     * @param WP_Post|WP_Term $flexible_content_object
     * @param string $builder_name
     */
    $flexible_content_object = apply_filters( 'raptor_flexible_content_object', $flexible_content_object, $builder_name );

    /**
     * Filter the field selector passed into `have_rows()`.
     * 
     * @deprecated Use `raptor_flexi_field_key` filter instead.
     * 
     * @param string The default selector with `$builder_name` dynamically added as a suffix.
     * @param WP_Post|WP_Term $flexible_content_object
     */
    $field_selector = apply_filters( 'raptor_flexi_blocks_builder_field_selector', raptor_get_flexi_field_key(), $flexible_content_object );
    
    if ( have_rows( $field_selector, $flexible_content_object ) ) {
        echo '<div class="flexi-blocks-builder builder-' . $builder_name . '">';

        /**
         * Fires before the builder loop
         */
        do_action( 'raptor_flexi_loop_start', $builder_name );

        while ( have_rows( $field_selector, $flexible_content_object ) ) {
            the_row();
            raptor_setup_flexi_block_data();

            $block = get_row_layout();

            if ( !$block ) {
                continue;
            }

            $settings = raptor_get_block_settings();
            $wrapper_attr = raptor_get_block_wrapper_attributes( $block, $settings );

            if ( is_flexi_block_hidden() ) {
                continue;
            }

            if ( Utils\block_exists( $block ) ) {
                echo '<section ' . raptor_output_attr( $wrapper_attr ) . '>';

                /**
                 * Fires before the block contents
                 */
                do_action( 'raptor_flexi_open_block', $block );

                /**
                 * Retrieve the block template
                 */
                raptor_get_block( $block, [ 'settings' => $settings ] );

                /**
                 * Fires after the block contents
                 */
                do_action( 'raptor_flexi_close_block', $block );

                echo '</section>';
            }
        }

        /**
         * Fires after the builder loop
         */
        do_action( 'raptor_flexi_loop_end', $builder_name );

        echo '</div>';
        
    } else {
        /**
         * Fires when no blocks are found
         */
        do_action( 'raptor_flexi_no_blocks_found', $builder_name );
    }
}


/**
 * Load one or more Global Flexi Blocks directly into a template.
 * 
 * @param int|array[int] $global_block_ids The global block ID's
 */
function raptor_global_flexi_blocks( $global_block_ids ) {
    global $flexi_blocks;
    $_flexi_blocks = [];

    echo '<div class="flexi-blocks-builder global-blocks">';

    /**
     * Fires before the builder loop
     */
    do_action( 'raptor_flexi_loop_start', false );

    if ( is_int( $global_block_ids ) ) {
        $global_block_ids = [ $global_block_ids ];
    }

    foreach ( $global_block_ids as $global_block_id ) {
        $flexi_block = raptor_setup_flexi_block_data( $global_block_id );
        $_flexi_blocks[] = $flexi_block;

        $block = $flexi_block['acf_fc_layout'];
        $settings = raptor_get_block_settings();
        $wrapper_attr = raptor_get_block_wrapper_attributes( $block, $settings );
    
        if ( Utils\block_exists( $block ) ) {
            echo '<section ' . raptor_output_attr( $wrapper_attr ) . '>';
    
            /**
             * Fires before the block contents
             */
            do_action( 'raptor_flexi_open_block', $block );
    
            /**
             * Retrieve the block template
             */
            raptor_get_block( $block, [ 'settings' => $settings ] );
    
            /**
             * Fires after the block contents
             */
            do_action( 'raptor_flexi_close_block', $block );
    
            echo '</section>';
        }
    }

    if ( !$flexi_blocks ) {
        $flexi_blocks = $_flexi_blocks;
    }

    /**
     * Fires after the builder loop
     */
    do_action( 'raptor_flexi_loop_end', false );

    echo '</div>';
}


/**
 * Load one or more Global Flexi Blocks directly into a template.
 * 
 * @param string $block_name The name of the Flexi Block
 */
function raptor_preview_flexi_block( string $block_name ) {
    global $raptor_blocks_library;

    if ( !isset( $raptor_blocks_library[ $block_name ] ) ) {
        return;
    }

    $block = $raptor_blocks_library[ $block_name ];
    $fields = $block->example['fields'];
    $settings = isset( $block->example['settings'] ) ? $block->example['settings'] : [];

    foreach ( $fields as $name => &$value ) {
        $field_index = array_search( $name, array_column( $block->fields, 'name' ) );
        $field = $block->fields[ $field_index ];

        if ( $field['type'] === 'wysiwyg' ) {
            $value  = apply_filters( 'the_content', $value );
        }

        if ( in_array( $field['type'], [ 'repeater', 'group' ] ) ) {
            $value = array_map( function( $row ) use( $field ) {
                foreach ( $row as $name => &$sub_value ) {
                    $field_index = array_search( $name, array_column( $field['sub_fields'], 'name' ) );
                    $sub_field = $field['sub_fields'][ $field_index ];
        
                    if ( $sub_field['type'] === 'wysiwyg' ) {
                        $sub_value = apply_filters( 'the_content', $sub_value );
                    }
                }

                return $row;
            }, $value );
        }
    }

    if ( empty( $settings ) && isset( $block->args['settings'] ) ) {
        $settings = array_map( function( $setting ) {
            return isset( $setting['default_value'] ) ? $setting['default_value'] : false;
        }, $block->args['settings'] );
    }

    $block_data = array_merge(
        [ 'acf_fc_layout' => $block_name ],
        $fields,
        [
            'settings' => $settings,
            'block_before' => isset( $block->example['block_before'] ) ? apply_filters( 'the_content', $block->example['block_before'] ) : '',
            'block_after' => isset( $block->example['block_after'] ) ? apply_filters( 'the_content', $block->example['block_after'] ) : ''
        ]
    );

    $block_data = raptor_preview_flexi_block_format_data( $block_data, $block_name );
    
    /**
     * Load data from a global block instead if `global_id` parameter is set.
     */
    if ( get_query_var( 'flexi_global_id' ) ) {
        $global_id = get_query_var( 'flexi_global_id' );
        $global_post = get_post( $global_id );

        if ( $global_post ) {
            $block_data = Utils\get_global_block_data( $global_id );
            // Add back the global block ID so it can be accessed if needed.
            $block_data['settings']['use_global'] = $global_id;
        }
    }

    $GLOBALS['flexi_block'] = $block_data;
    $GLOBALS['flexi_blocks'] = [ $block_data ];

    echo '<div class="flexi-blocks-builder preview-block">';

    /**
     * Fires before the builder loop
     */
    do_action( 'raptor_flexi_loop_start' );

    $settings = raptor_get_block_settings();
    $wrapper_attr = raptor_get_block_wrapper_attributes( $block_name, $settings );

    if ( Utils\block_exists( $block_name ) ) {
        echo '<section ' . raptor_output_attr( $wrapper_attr ) . '>';

        /**
         * Fires before the block contents
         */
        do_action( 'raptor_flexi_open_block', $block_name );

        /**
         * Retrieve the block template
         */
        raptor_get_block( $block_name, [ 'settings' => $settings ] );

        /**
         * Fires after the block contents
         */
        do_action( 'raptor_flexi_close_block', $block_name );

        echo '</section>';
    }

    /**
     * Fires after the builder loop
     */
    do_action( 'raptor_flexi_loop_end' );

    echo '</div>';
}


/**
 * Load all Flexi blocks in the site library.
 */
function raptor_flexi_block_glossary() {
    global $raptor_blocks_library;

    $flexi_blocks = array_map( function( $flexi_block ) {
        $fields = $flexi_block->example['fields'];

        foreach ( $fields as $name => &$value ) {
            $field_index = array_search( $name, array_column( $flexi_block->fields, 'name' ) );
            $field = $flexi_block->fields[ $field_index ];

            if ( $field['type'] === 'wysiwyg' ) {
                $value  = apply_filters( 'the_content', $value );
            }

            if ( in_array( $field['type'], [ 'repeater', 'group' ] ) ) {
                $value = array_map( function( $row ) use( $field ) {
                    foreach ( $row as $name => &$sub_value ) {
                        $field_index = array_search( $name, array_column( $field['sub_fields'], 'name' ) );
                        $sub_field = $field['sub_fields'][ $field_index ];
            
                        if ( $sub_field['type'] === 'wysiwyg' ) {
                            $sub_value = apply_filters( 'the_content', $sub_value );
                        }
                    }

                    return $row;
                }, $value );
            }
        }

        $settings = isset( $flexi_block->example['settings'] ) ? $flexi_block->example['settings'] : [];

        if ( empty( $settings ) && isset( $flexi_block->args['settings'] ) ) {
            $settings = array_map( function( $setting ) {
                return isset( $setting['default_value'] ) ? $setting['default_value'] : false;
            }, $flexi_block->args['settings'] );
        }

        return array_merge(
            [ 'acf_fc_layout' => $flexi_block->name ],
            $fields,
            [
                'settings' => $settings,
                'block_before' => isset( $flexi_block->example['block_before'] ) ? apply_filters( 'the_content', $flexi_block->example['block_before'] ) : '',
                'block_after' => isset( $flexi_block->example['block_after'] ) ? apply_filters( 'the_content', $flexi_block->example['block_after'] ) : ''
            ]
        );
    }, $raptor_blocks_library );

    $GLOBALS['flexi_blocks'] = $flexi_blocks;

    echo '<div class="flexi-blocks-builder">';

    /**
     * Fires before the builder loop
     */
    do_action( 'raptor_flexi_loop_start' );

    foreach ( $flexi_blocks as $flexi_block ) {
        $GLOBALS['flexi_block'] = $flexi_block;
        $block_name = $flexi_block['acf_fc_layout'];

        $settings = raptor_get_block_settings();
        $wrapper_attr = raptor_get_block_wrapper_attributes( $block_name, $settings );
    
        if ( Utils\block_exists( $block_name ) ) {
            echo '<section ' . raptor_output_attr( $wrapper_attr ) . '>';
    
            /**
             * Fires before the block contents
             */
            do_action( 'raptor_flexi_open_block', $block_name );
    
            /**
             * Retrieve the block template
             */
            raptor_get_block( $block_name, [ 'settings' => $settings ] );
    
            /**
             * Fires after the block contents
             */
            do_action( 'raptor_flexi_close_block', $block_name );
    
            echo '</section>';
        }
    }

    /**
     * Fires after the builder loop
     */
    do_action( 'raptor_flexi_loop_end' );

    echo '</div>';
}

/**
 * Fill in the block settings so each field index is set.
 * 
 * @param array $data
 * @param string $block_name
 * @return array
 */
function raptor_preview_flexi_block_format_data( array $data, string $block_name ) {
    global $raptor_blocks_library;

    foreach ( $raptor_blocks_library[ $block_name ]->fields as $field ) {
        if ( $field['name'] == 'settings' ) {
            $settings = [];

            foreach ( $field['sub_fields'] as $sub_field ) {
                if ( !isset( $data['settings'][ $sub_field['name'] ] ) ) {
                    $data['settings'][ $sub_field['name'] ] = isset( $sub_field['default_value'] ) ? $sub_field['default_value'] : false;
                }
            }
        }
    }

    $processed_fields = [];
    $processed_settings = [];
    $acf_post_id = 'raptor_flexi_block_example-' . $block_name;

    foreach ( $raptor_blocks_library[ $block_name ]->fields as $field ) {
        if ( isset( $data['fields'][ $field['name'] ] ) ) {
            $example_value = $data['fields'][ $field['name'] ];

            $field['_name'] = $field['name'];
            /**
             * ACF expects `_name` to be set on repeater sub fields.
             */
            if ( isset( $field['sub_fields'] ) ) {
                foreach ( $field['sub_fields'] as &$_sub_field ) {
                    $_sub_field['_name'] = $_sub_field['name'];
                }

                $sub_field_values = [];

                if ( $field['type'] === 'repeater' ) {
                    for ( $i = 0; $i < count( $example_value ); $i++ ) {
                        foreach ( $field['sub_fields'] as $sub_field ) {
                            $sub_field_values[$i][ $sub_field['key'] ] = $example_value[$i][ $sub_field['name'] ];
                        }
                    }

                } else if ( $field['type'] === 'group' ) {
                    foreach ( $field['sub_fields'] as $sub_field ) {
                        $sub_field_values[ $sub_field['key'] ] = $example_value[ $sub_field['name'] ];
                    }
                }

                $example_value = $sub_field_values;
            }

            $processed_fields[ $field['name'] ] = acf_format_value( $example_value, $acf_post_id, $field );
        }

        if ( $field['name'] === 'settings' ) {
            $setting_field_values = [];

            foreach ( $field['sub_fields'] as &$_sub_field ) {
                $_sub_field['_name'] = $_sub_field['name'];

                $setting_field_values[ $_sub_field['key'] ] = $data['settings'][ $_sub_field['name'] ];
            }

            $processed_settings = acf_format_value( $setting_field_values, $acf_post_id, $field );
        }

        if ( $field['name'] === 'block_before' && isset( $data['block_before'] ) && !empty( $data['block_before'] ) ) {
            $data['block_before'] = acf_format_value( $data['block_before'], $acf_post_id, $field );
        }

        if ( $field['name'] === 'block_after' && isset( $data['block_after'] ) && !empty( $data['block_after'] ) ) {
            $data['block_after'] = acf_format_value( $data['block_after'], $acf_post_id, $field );
        }
    }

    if ( !empty( $processed_fields ) ) {
        $data['fields'] = $processed_fields;
    }

    if ( !empty( $processed_settings ) ) {
        $data['settings'] = $processed_settings;
    }

    return $data;
}


/**
 * Using the block settings, build the attributes of the
 * block wrapper element. The attributes help identify and target
 * the block instance.
 * 
 * @param string $block_name The block name
 * @param array $settings The block settings
 * @return array The attributes for the block wrapper element
 */
function raptor_get_block_wrapper_attributes( string $block_name, array $settings = [] ) {
    $attributes = [ 'class' => 'flexi-block block--' . $block_name ];

    foreach ( $settings as $setting => $value ) {
        if ( $setting == 'use_global' ) {
            continue;
        }

        if ( preg_match( '/<svg/', $value ) ) {
            continue;
        }
        
        if ( $setting == 'id' ) {
            $attributes['id'] = $value;

        } else if ( is_bool( $value ) ) {
            $attributes['class'] .= $value ? " {$setting}" : '';
            
        } else {
            /**
             * Control if a setting should have a class name added based on its name and value.
             * 
             * @param bool
             * @param string $setting
             * @param string $value
             */
            if ( apply_filters( 'raptor_flexi_block_wrapper_ignore_setting', $setting === 'background' && $value === 'white', $setting, $value ) ) {
                continue;
            } elseif ( $setting == 'background' ) {
                $setting = 'bg';

                $attributes['class'] .= ' has-bg';
            }

            $attributes['class'] .= " {$setting}-{$value}";
        }                 
    }
    $attributes['class'] .= is_flexi_block_global() ? ' global_block global_block-' . $settings['use_global'] : '';

    return apply_filters( 'raptor_get_block_wrapper_attributes', $attributes, $block_name, $settings );
}


/**
 * Print the html of a block heading element
 * 
 * @param string $content Use html string instead of active field
 * @param string The name of the ACF field
 */
function raptor_block_heading( $field = '' ) {
    if ( empty( $field ) ) {
        $field = raptor_acf_get_field_selector( 'block_before' );
    }

    $content = get_flexi_field( $field );

    if ( $content ) {
        ?>
<div class="block-heading">
        <?php echo $content; ?>
</div>
        <?php
    }
}


/**
 * Print the html of a block after content element
 * 
 * @param string $content Use html string instead of active field
 * @param string The name of the ACF field
 */
function raptor_block_after( $content = '', $field = '' ) {
    if ( empty( $field ) ) {
        $field = raptor_acf_get_field_selector( 'block_after' );
    }

    $content = get_flexi_field( $field );

    if ( $content ) {
        ?>
<div class="block-after">
        <?php echo $content; ?>
</div>
        <?php
    }
}


/**
 * Get the current block settings via the ACF group
 * 
 * @return array The settings of the block
 */
function raptor_get_block_settings() {
    global $flexi_block;

    return $flexi_block['settings'];
}


/**
 * Get a field value within the context of a Flexi Block, where Global Blocks are supported.
 * 
 * @param string $selector The field selector
 * @param bool $format_value Should format the field value
 */
function get_flexi_field( string $selector, bool $format_value = true ) {
    global $flexi_block;

    if ( isset( $flexi_block[ $selector ] ) ) {
        return $flexi_block[ $selector ];
    }

    return false;
}


/**
 * @param string $selector
 * @param bool $format_value
 */
function the_flexi_field( string $selector, bool $format_value = true ) {
    $value = get_flexi_field( $selector, $format_value );

    if ( !$value ) {
        return null;
    }

    echo $value;
}


/**
 * Find out if current flexi block is a global one.
 * 
 * @return int|false The global block ID if set, or false.
 */
function is_flexi_block_global() {
    global $flexi_block;

    return isset( $flexi_block['settings']['use_global'] ) ? intval( $flexi_block['settings']['use_global'] ) : false;
}


/**
 * Check if a block is set as hidden.
 * 
 * @return bool
 */
function is_flexi_block_hidden() {
    global $flexi_block;
    
    return apply_filters( 'raptor_is_flexi_block_hidden', $flexi_block['settings']['hide'], $flexi_block );
}


/**
 * Find out if a page or term has a Flexi block.
 * 
 * @param string $block_name
 * @return bool
 */
function has_flexi_block( $block_names ) {
    global $flexi_blocks;

    $found = false;

    if ( gettype( $block_names ) == 'string' ) {
        $block_names = [ $block_names ];
    }

    foreach ( $flexi_blocks as $block ) {
        // Always respect hidden blocks.
        if ( isset( $block['settings']['hide'] ) && $block['settings']['hide'] ) {
            continue;
        }

        if ( in_array( $block['acf_fc_layout'], $block_names ) ) {
            $found = true;
        }
    }

    return $found;
}


/**
 * Generate ACF fields based on post types registed via Raptor.
 * 
 * @param string $format post_object|array
 * @return array
 */
function raptor_flexi_get_available_post_types( string $format = 'array' ) {
    global $raptor_post_types;

    $return = [];

    $post_types = $raptor_post_types;

    if ( function_exists( 'shiftr' ) ) {
        global $shiftr_post_types;

        $post_types = array_merge( $raptor_post_types, $shiftr_post_types );
    }

    foreach ( $post_types as $post_type => $instance ) {

        if ( $format == 'array' ) {
            $return[ $post_type ] = $instance->plural;

        } else if ( $format == 'post_object' ) {
            $return[] = Field_Types\post_object_field(
                $instance->plural,
                [
                    'key'       => 'raptor_' . $post_type . '_page_id',
                    'name'      => 'raptor_' . $post_type . '_page_id',
                    'post_type' => [ 'page' ],
                    'multiple'  => 0
                ]
            );
        }
    }

    return $return;
}


/**
 * Setup the data for a block in a loop (currently global only).
 * 
 * @param int $block_id
 */
function raptor_setup_flexi_block_data( int $global_block_id = 0 ) {
    unset( $GLOBALS['flexi_block'] );

    $flexi_block = get_row( true );

    if ( isset( $flexi_block['settings']['use_global'] ) && $flexi_block['settings']['use_global'] ) {
        $global_block_id = intval( $flexi_block['settings']['use_global'] );
    }

    if ( $global_block_id ) {
        $flexi_block = Utils\get_global_block_data( $global_block_id );
        // Add back the global block ID so it can be accessed if needed.
        $flexi_block['settings']['use_global'] = $global_block_id;
    }

    $GLOBALS['flexi_block'] = $flexi_block;

    return $flexi_block;
}


/**
 * Get the list of script dependancies for a Flexi block.
 * 
 * @param string $block_name
 * @return array|bool
 */
function get_flexi_block_scripts( string $block_name ) {
    global $raptor_blocks_library;

    $block_object = $raptor_blocks_library[ $block_name ];

    if ( !empty( $block_object->args['scripts'] ) ) {
        return $scripts;
    }

    return false;
}


/**
 * Get a list of Flexi blocks with a given script dependancy.
 * 
 * @param string $script_handle
 * @return array
 */
function get_flexi_blocks_with_script( string $script_handle ) {
    global $raptor_blocks_library;

    $blocks = [];

    foreach ( $raptor_blocks_library as $block_name => $block ) {
        if ( in_array( $script_handle, $block->args['scripts'] ) ) {
            $blocks[] = $block_name;
        }
    }

    return $blocks;
}


/**
 * Take ACF layout data and inject global block data.
 * 
 * @param array $data
 * @return array 
 */
function raptor_flexi_parse_global_blocks_data_formatted( array $data ) {
    return array_map( function( $block_data ) {
        $block_name = $block_data['acf_fc_layout'];

        if ( isset( $block_data['settings'] ) ) {
            $global_block_id = isset( $block_data['settings']['use_global'] ) ?
                intval( $block_data['settings']['use_global'] ) : false;

            if ( $global_block_id ) {
                $global_block = get_field( 'flexi_blocks_builder-global', $global_block_id );

                return $global_block[0];
            }
        }

        return $block_data;
    }, $data );
}
