<?php
namespace Raptor\ACF\Admin;

use Raptor\ACF\Field_Types;

add_action( 'acf/input/admin_enqueue_scripts', function() {
    wp_register_script( 'raptor-admin-acf', raptor_get_asset_url( 'assets/js/acf.js' ), [ 'acf-input', 'raptor-admin-script', 'wp-tinymce' ], raptor()->version, [ 'in_footer' => true ] );
    wp_enqueue_script( 'raptor-admin-acf' );

    wp_register_style( 'raptor-admin-acf', raptor_get_asset_url( 'assets/css/acf.css' ), [], raptor()->version );
    wp_enqueue_style( 'raptor-admin-acf' );
});


/**
 * Register Flexi Templates post type.
 */
new \Raptor_Custom_Post_Type(
    [
        'name' => 'flexi_template',
        'label' => 'Template',
        'menu_position' => 54,
        'menu_icon' => 'dashicons-block-default'
    ],
    [
        'supports' => [ 'title', 'author' ]
    ]
);


add_filter( 'raptor_custom_post_type_register_args', function( $args, $post_type ) {
    if ( $post_type == 'flexi_template' ) {
        $args['labels']['name'] = 'Flexi';
        $args['labels']['all_items'] = 'All Templates';
        $args['labels']['add_new_item'] = 'New Template';
    }

    return $args;
}, 10, 2 );


/**
 * Register Flexi global blocks
 */
new \Raptor_Custom_Post_Type(
    [
        'name' => 'flexi_global',
        'label' => 'Global Block',
        'menu_icon' => 'dashicons-block-default'
    ],
    [
        'supports' => [ 'title', 'author' ],
        'show_in_menu'  => 'edit.php?post_type=flexi_template',
    ]
);


add_filter( 'raptor_custom_post_type_register_args', function( $args, $post_type ) {
    if ( $post_type == 'flexi_global' ) {
        $args['labels']['all_items'] = 'Global Blocks';
    }

    return $args;
}, 10, 2 );


/**
 * Apply Flexi Page Template
 */
add_filter( 'acf/load_value/type=flexible_content', function( $value, $post_id, $field ) {
    if ( $value !== null || !is_admin() ) {
        return $value;
    }
    /**
     * Control whether should attempt to load Flexi Template.
     * 
     * @param bool
     * @param array $field
     */
    if ( apply_filters( 'raptor_flexi_template_skip_load', ( !preg_match( '/flexi/', $field['name'] ) || $field['name'] == 'flexi_blocks_builder-global' ), $field ) ) {
        return $value;
    }

    $requested_template = isset( $_REQUEST['flexi_template'] ) ? intval( $_REQUEST['flexi_template'] ) : 0;

    if ( $requested_template ) {
        $flexi_template_post = \get_post( $requested_template );

    } else {
        /**
         * Must not be nullable value, as this causes the most recent Flexi Template
         * to be applied when a post/term is not found.
         */
        $auto_apply_value = '0';
        
        if ( is_int( $post_id ) ) {
            $post = \get_post( $post_id );
            $auto_apply_value = $post->post_type;
        }

        if ( preg_match( '/^term_0$/', $post_id ) ) {
            $screen = get_current_screen();
            $auto_apply_value = $screen->taxonomy;
        }

        $templates = new \WP_Query([
            'post_type' => 'flexi_template',
            'meta_key' => 'auto_apply',
            'meta_value' => $auto_apply_value,
            'posts_per_page' => 1
        ]);

        $flexi_template_post = empty( $templates->posts ) ? false : $templates->posts[0];
    }

    if ( !$flexi_template_post ) {
        return $value;
    }

    return \get_field(
        raptor_get_flexi_field_key(),
        $flexi_template_post->ID,
        false
    );
}, 100, 3 );


/**
 * Add the block slug as a post meta value for easy querying.
 */
add_action( 'save_post_flexi_global', function( int $post_ID, \WP_Post $post, bool $update ) {
    if ( isset( $_POST['acf'] ) ) {
        /**
         * The way ACF names the row indexes varies between post publish and update. So reset the keys for the array.
         */
        $global_blocks_builder = array_values( $_POST['acf']['flexi_blocks_builder-global'] );
        $block = $global_blocks_builder[0]['acf_fc_layout'];

        update_post_meta( $post_ID, 'flexi_block', $block );
    }
}, 10, 3 );


/**
 * Add Block to columns.
 */
add_filter( 'manage_flexi_global_posts_columns', function( $columns ) {
    return array_merge(
        array_slice( $columns, 0, 2 ),
        [ 'flexi_block' => 'Block' ],
        array_slice( $columns, 2 )
    );
});


add_filter( 'manage_edit-flexi_global_sortable_columns', function( $columns ) {
    $columns['flexi_block'] = 'flexi_block';

    return $columns;
});


add_action( 'manage_flexi_global_posts_custom_column', function( $column, $post_id ) {
    global $raptor_blocks_library;

    if ( $column == 'flexi_block' ) {
        $title = strval( $post_id );
        $flexi_block_name = get_post_meta( $post_id, 'flexi_block', true );

        if ( isset( $raptor_blocks_library[ $flexi_block_name ] ) ) {
            echo $raptor_blocks_library[ $flexi_block_name ]->label;
        } else {
            echo $flexi_block_name;
        }
    }
}, 10, 2 );


add_filter( 'query_vars', function( $query_vars ) {
    $query_vars[] = 'flexi_block';

    return $query_vars;
}, 10, 1 );


add_action( 'pre_get_posts', function( $query ) {
    if ( !is_admin() ) {
        return;
    }

    $orderby = $query->get( 'orderby' );

    if ( $orderby == 'flexi_block' ) {
        $query->set( 'meta_key', 'flexi_block' );
        $query->set( 'orderby', 'meta_value' );
        $query->set( 'order', 'ASC' );
    }
});


add_action( 'admin_menu', function() {
    global $menu;

    $menu[] = [ '', 'read', 'separator-raptor', '', 'wp-menu-separator raptor' ];
});


add_filter( 'custom_menu_order', '__return_true' );


add_filter( 'menu_order', function( $menu_order ) {
    $raptor_separator = array_search( 'separator-raptor', $menu_order, true );
    $raptor_flexi_template = array_search( 'edit.php?post_type=flexi_template', $menu_order, true );
    $raptor_acf_options = array_search( 'site-options', $menu_order, true );

    $raptor_menu_order = [];

    foreach ( $menu_order as $index => $item ) {

        if ( 'edit.php?post_type=flexi_template' === $item ) {
            $raptor_menu_order[] = 'separator-raptor';
            $raptor_menu_order[] = $item;
            $raptor_menu_order[] = 'edit.php?post_type=raptor_form';
            $raptor_menu_order[] = 'site-options';
            unset( $menu_order[ $raptor_separator ] );
            unset( $menu_order[ $raptor_flexi_template ] );
            unset( $menu_order[ $raptor_acf_options ] );

        } elseif ( !in_array( $item, [ 'separator-raptor' ], true ) ) {
            $raptor_menu_order[] = $item;
        }
    }

    return $raptor_menu_order;
}, 10, 1 );


/**
 * Save an existing page as a Flexi Template.
 */
function save_existing_as_flexi_template( int $post_ID, \WP_Post $post, bool $update ) {
    $template_name = isset( $_POST['flexi_template_name'] ) ? sanitize_text_field( $_POST['flexi_template_name'] ) : '';

    if ( !$template_name ) {
        return;
    }
    
    if ( !wp_is_post_revision( $post_ID ) && $post->post_type != 'flexi_template' ) {
        $flexi_field_key = raptor_get_flexi_field_key();

        $template_id = wp_insert_post([
            'post_title' => $template_name,
            'post_type' => 'flexi_template',
            'post_status' => 'publish'
        ]);

        $template_data = \get_field(
            $flexi_field_key,
            $post_ID,
            false
        );

        update_field( $flexi_field_key, $template_data, $template_id );

        wp_redirect( get_edit_post_link( $template_id, '' ) );
        exit;
    }
}
add_action( 'save_post', 'Raptor\ACF\Admin\save_existing_as_flexi_template', 10, 3 );


/**
 * Add export action to Flexi Template posts.
 */
add_filter( 'post_row_actions', function( array $actions, \WP_Post $post ) {
    global $wp;

    if ( $post->post_type == 'flexi_template' && $post->post_status == 'publish' ) {
        $actions['export'] = sprintf( '<a href="?post_type=flexi_template&export-flexi-template=%s">Export</a>', $post->ID );
    }

    return $actions;
}, 100, 2 );


/**
 * Export a Flexi Template to a JSON file.
 * 
 * @param int $flexi_template_id
 */
function raptor_flexi_template_export( int $flexi_template_id ) {
    $fields_data = get_fields( $flexi_template_id, false );
    $post = get_post( $flexi_template_id );

    $json = [
        'title' => $post->post_title,
        'flexi_data' => raptor_flexi_parse_global_blocks_data( $fields_data[raptor_get_flexi_field_key()] ),
        'auto_apply' => isset( $fields_data['auto_apply'] ) ? $fields_data['auto_apply'] : false
    ];

    $file_name = 'flexi-template-export-' . date( 'Y-m-d' ) . '.json';

    header( 'Content-Description: File Transfer' );
    header( "Content-Disposition: attachment; filename={$file_name}" );
    header( 'Content-Type: application/json; charset=utf-8' );

    echo wp_json_encode( $json );
    die;
}


/**
 * Trigger Flexi Template export function.
 */
add_action( 'admin_init', function() {
    $flexi_template_id = isset( $_GET['export-flexi-template'] ) ? intval( $_GET['export-flexi-template'] ) : false;

    if ( $flexi_template_id ) {
        raptor_flexi_template_export( $flexi_template_id );
    }
});


 
function raptor_flexi_template_tools_page_setup() {
    add_submenu_page(
        'edit.php?post_type=flexi_template',
        'Tools',
        'Tools',
        'manage_options',
        'flexi-tools',
        'Raptor\ACF\Admin\raptor_flexi_template_tools_page'
    );
}
add_action( 'admin_menu', 'Raptor\ACF\Admin\raptor_flexi_template_tools_page_setup' );


function raptor_flexi_template_import( $file ) {
    // Check for errors.
    if ( $file['error'] ) {
        return false;
    }

    // Check file type.
    if ( pathinfo( $file['name'], PATHINFO_EXTENSION ) !== 'json' ) {
        return false;
    }

    $json = file_get_contents( $file['tmp_name'] );
    $json = json_decode( $json, true );

    $flexi_template_id = wp_insert_post([
        'post_type' => 'flexi_template',
        'post_title' => $json['title'],
        'post_status' => 'publish'
    ]);

    update_field( raptor_get_flexi_field_key(), $json['flexi_data'], $flexi_template_id );
    update_field( 'auto_apply', $json['auto_apply'], $flexi_template_id );

    /**
     * After success, redirect to post edit page.
     */
    wp_redirect( get_edit_post_link( $flexi_template_id, '' ) );
    exit;
}


function raptor_flexi_template_import_meta_box() {
    ?>
    <form method="post" enctype="multipart/form-data" class="raptor-form">
        <p>Select the Flexi Template JSON file you would like to import. When you click the import button below, Raptor will import the template.</p>
        <div class="raptor-field field-type--file">
            <label for="flexi_template_json">Import JSON file</label>
            <input type="file" name="flexi_template_json" id="flexi_template_json" required> 
        </div>
        <button type="submit" class="button button-primary" name="flexi_template_import" value="import">Import</button>
    </form>
    <?php
}


function raptor_flexi_template_tools_page() {
    $screen = get_current_screen();

    if ( isset( $_FILES['flexi_template_json'] ) ) {
        raptor_flexi_template_import( $_FILES['flexi_template_json'] );
    }

    add_meta_box(
        'raptor_flexi_template_tool_import',
        'Import Flexi Template',
        'Raptor\ACF\Admin\raptor_flexi_template_import_meta_box',
        $screen->id,
        'normal'
    );

    ?>
    <div class="wrap" id="raptor-admin-tools">
        <h1>Tools</h1>
        
        <div class="raptor-meta-box-wrap">
            <?php do_meta_boxes( $screen->id, 'normal', '' ); ?>	
        </div>
    </div>
    <?php
}


/**
 * Take raw format Flexi Template data and inject global block data.
 * 
 * @param array $data
 * @return array 
 */
function raptor_flexi_parse_global_blocks_data( array $data, bool $format = false ) {
    return array_map( function( $block_data ) use( $format ) {
        if ( isset( $block_data['settings'] ) ) {
            $global_block_id = isset( $block_data['settings']['use_global'] ) ?
                intval( $block_data['settings']['use_global'] ) : false;

            if ( $global_block_id ) {
                $global_block = get_field( 'flexi_blocks_builder-global', $global_block_id, $format );

                return $global_block[0];
            }
        }

        return $block_data;
    }, $data );
}


add_action( 'wp_ajax_raptor_get_flexi_templates', function() {
    $template_posts = new \WP_Query([
        'post_type' => 'flexi_template',
        'posts_per_page' => -1
    ]);

    if ( $template_posts->have_posts() ) {
        $templates = array_map( function( $post ) {
            return [
                'title' => $post->post_title,
                'id' => $post->ID
            ];
        }, $template_posts->posts );

        wp_send_json([
            'templates' => $templates
        ]);
    }

    wp_send_json([
        'error' => 'No Flexi Templates found.'
    ]);
});


add_action( 'wp_ajax_raptor_get_global_blocks', function() {
    $global_posts = new \WP_Query([
        'post_type' => 'flexi_global',
        'posts_per_page' => -1
    ]);

    if ( $global_posts->have_posts() ) {
        $blocks = array_map( function( $post ) {
            return [
                'title' => $post->post_title,
                'id' => $post->ID,
                'block' => get_post_meta( $post->ID, 'flexi_block', true ),
                'lastModDate' => strtotime( $post->post_modified )
            ];
        }, $global_posts->posts );

        wp_send_json([
            'blocks' => $blocks
        ]);
    }

    wp_send_json([
        'error' => 'No global blocks found.'
    ]);
});


/**
 * Don't show current post as option for recent posts.
 */
add_filter( 'acf/fields/post_object/query/name=related_posts', function( $args, $field, $post_id ) {
    $args['post__not_in'] = [ $post_id ];

    return $args;
}, 10, 3 );


/**
 * Save Block as Global Block
 */
add_action( 'wp_ajax_raptor_save_as_global_block', function() {
    $post_id = $_POST['post_id'];
    $block_index = $_POST['block_index'];
    $global_block_title = $_POST['global_block_title'];
    /**
     * Get the Flexi Blocks of the post.
     */
    $post_flexi_blocks_builder = \get_field(
        raptor_get_flexi_field_key(),
        $post_id,
        false
    );
    /**
     * Get the Block to clone.
     */
    $block = $post_flexi_blocks_builder[ $block_index ];
    /**
     * Create Global Block.
     */
    $global_block_id = wp_insert_post([
        'post_title' => sanitize_text_field( $global_block_title ),
        'post_type' => 'flexi_global',
        'post_status' => 'publish'
    ]);
    /**
     * Save Block data to Global Block.
     */
    update_field( 'flexi_blocks_builder-global', [ $block ], $global_block_id );
    update_post_meta( $global_block_id, 'flexi_block', $block['acf_fc_layout'] );
    /**
     * Send back response with new Global Block ID
     */
    wp_send_json([
        'global_block_id' => $global_block_id
    ]);
});



/**
 * Duplicate a Global Block
 */
add_action( 'wp_ajax_raptor_duplicate_global_block', function() {
    $post_id = $_POST['post_id'];

    /**
     * Get the Flexi Blocks of the post.
     */
    $source_builder = \get_field(
        'flexi_blocks_builder-global',
        $post_id,
        false
    );
    
    /**
     * Create Global Block.
     */
    $global_block_id = wp_insert_post([
        'post_title' => get_the_title( $post_id ) . ' (copy)',
        'post_type' => 'flexi_global',
        'post_status' => 'publish'
    ]);
    /**
     * Save Block data to Global Block.
     */
    update_field( 'flexi_blocks_builder-global', $source_builder, $global_block_id );
    update_post_meta( $global_block_id, 'flexi_block', $source_builder[0]['acf_fc_layout'] );
    /**
     * Send back response with new Global Block ID
     */
    wp_send_json([
        'global_block_id' => $global_block_id
    ]);
});


add_action( 'post_row_actions', function( array $actions, \WP_Post $post ) {
    if ( $post->post_type !== 'flexi_global' ) {
        return $actions;
    }

    $actions['duplicate'] = sprintf( '<a href="?raptor-duplicate-global-block=%s">Duplicate</a>', $post->ID );

    return $actions;
}, 10, 2 );


/**
 * Convert the existing block to use the new global block.
 * 
 * Priority 5 is ran before actual save.
 * 
 * @param int|string $post_id
 */
add_action( 'acf/save_post', function( $post_id ) {
    if ( !isset( $_POST['raptor_save_as_global_block'] ) ) {
        return;
    }

    list( $global_block_id, $old_layout_row_index, $new_layout_row_index ) = explode( '|', $_POST['raptor_save_as_global_block'] );

    $flexi_field_key = raptor_get_flexi_field_key();

    if ( isset( $_POST['acf'][$flexi_field_key] ) ) {
        $layout = $_POST['acf'][$flexi_field_key]["row-$old_layout_row_index"];

        $layout_name = $layout['acf_fc_layout'];

        $_POST['acf'][$flexi_field_key]["row-$old_layout_row_index"]["{$layout_name}_settings"]["{$layout_name}_block_setting_use_global"] = $global_block_id;
    }
}, 5 );


/**
 * Save the block again to ensure it has up-to-date data.
 * 
 * @param int|string $post_id
 */
add_action( 'acf/save_post', function( $post_id ) {
    if ( !isset( $_POST['raptor_save_as_global_block'] ) ) {
        return;
    }

    list( $global_block_id, $old_layout_row_index, $new_layout_row_index ) = explode( '|', $_POST['raptor_save_as_global_block'] );

    $post_flexi_blocks_builder = \get_field(
        raptor_get_flexi_field_key(),
        $post_id,
        false
    );
    /**
     * Get the Block to clone.
     */
    $block = $post_flexi_blocks_builder[ $new_layout_row_index ];

    update_field( 'flexi_blocks_builder-global', [ $block ], $global_block_id );
}, 20 );


/**
 * Load existing Flexi Template
 */
add_action( 'acf/save_post', function( $post_id ) {
    if ( !isset( $_POST['raptor_load_flexi_template'] ) ) {
        return;
    }

    $flexi_field_key = raptor_get_flexi_field_key();
    $flexi_template_id = $_POST['raptor_load_flexi_template'];

    $post_flexi_blocks_builder = \get_field(
        $flexi_field_key,
        $flexi_template_id,
        false
    );

    update_field( $flexi_field_key, $post_flexi_blocks_builder, $post_id );
}, 20 );


/**
 * On save for Site Options, flush the rewrite rules.
 * 
 * We can't run `flush_rewrite_rules()` on post save, so we can use a transient so
 * on the next request (which is a redirect), we can check and flush the rules.
 *
 * @param mixed $post_id
 */
function save_options_rewrite_flush( $post_id ) {
    if ( $post_id == 'options' && isset( $_POST['acf']['page_setup'] ) ) {
        set_transient( 'raptor_flush_rewrite_rules', true, 5 );
    }
}
add_action( 'acf/save_post', 'Raptor\ACF\Admin\save_options_rewrite_flush' );


/**
 * Flush the rules
 */
function options_flush_rewrite_rules() {
    if ( get_transient( 'raptor_flush_rewrite_rules' ) ) {
        flush_rewrite_rules( false );
    }
}
add_action( 'admin_init', 'Raptor\ACF\Admin\options_flush_rewrite_rules' );


/**
 * Check if site_options group exists and replace Raptor Settings (raptor_settings_raptor)
 * 
 * @param array $groups The $raptor_groups_library global variable
 */
function check_site_options_exists( array $groups ) {
    $has_site_options = array_key_exists( 'site_options', $groups );

    if ( $has_site_options ) {
        /**
         * Unset doesn't seem to work... so remove the location value
         */
        if ( isset( $groups['raptor_settings_raptor'] ) ) {
            $groups['raptor_settings_raptor']->acf_group_data['location'][0][0]['value'] = '';
        }

        $groups['site_options']->acf_group_data['location'][0][0]['value'] = 'raptor-settings';
    }

    return $groups;
}
add_filter( 'raptor_acf_groups', 'Raptor\ACF\Admin\check_site_options_exists' );
