/*global raptor*/

export default class ConditionalLogic {
  /** @type {HTMLFormElement} */
  form = null

  /** @type {any[]} */
  fields = []

  /** @type {any[] | string} */
  groups = []

  /** @type {string} The class used to mark a field/group as hidden */
  className = 'conditionally-hidden'


  /**
   * @param {HTMLFormElement} form 
   */
  constructor(form) {
    const formName = form.getAttribute('name')
    this.form = form
    this.fields = raptor.forms[formName]?.fields
    this.groups = raptor.forms[formName]?.groups

    if (!this.fields) {
      console.warn('Cannot perform conditional logic, `fields` not found.')
      return
    }

    this.init()
  }


  init() {
    this.initFields()

    if (!Array.isArray(this.groups)) {
      return
    }
    
    this.initGroups()
  }


  initFields() {
    this.fields.forEach((field) => {
      const fieldEl = this.getFieldEl(this.form.elements[field.name])
      const fieldWrapper = fieldEl.parentElement

      if (field?.conditional_logic_enabled) {
        fieldWrapper.classList.add(this.className)
      }

      if (['select', 'checkbox', 'radio'].indexOf(field.type) >= 0) {
        fieldEl.addEventListener('change', () => {
          this.runConditionalLogic(field)
        })
      } else {
        fieldEl.addEventListener('blur', () => {
          this.runConditionalLogic(field)
        })
      }
    })
  }


  initGroups() {
    this.groups.forEach((group) => {
      const fieldset = this.getFieldEl(this.form[group.name])

      if (group?.conditional_logic_enabled) {
        fieldset.classList.add(this.className)
      }
    })
  }


  /**
   * @param {HTMLElement} el 
   * @returns {HTMLElement}
   */
  getFieldEl(el) {
    if (el.constructor === RadioNodeList) {
      return el[0]
    }

    return el
  }


  isElementHidden(object) {
    const conditions = object.conditional_logic.filter((orLogic) => {
      return orLogic.map((andLogic) => {
        const fieldValue = this.form.elements[andLogic.field].value

        if (andLogic.operator === '!=') {
          return fieldValue !== andLogic.value
        }

        return fieldValue === andLogic.value
      }).every((logic) => {
        return logic === true
      })
    })

    return conditions.length < 1
  }


  /**
   * @param {any} thing Field or Option object
   * @param {HTMLElement} el The field/option element
   */
  controlVisibility(thing, el) {
    const isOption = el.constructor === HTMLOptionElement
    let targetEl = [HTMLFieldSetElement, HTMLOptionElement]
      .indexOf(el.constructor) >= 0 ? el : el.parentElement

    if (thing.type === 'radio' || thing.type === 'checkbox') {
      targetEl = el.parentElement
    }

    if (this.isElementHidden(thing)) {
      targetEl.classList.add(this.className)

      if (isOption) {
        el.required = false
      }
    } else {
      targetEl.classList.remove(this.className)

      if (isOption) {
        el.required = true
      }
    }
  }


  /**
   * @param {any} fieldsOrGroups 
   * @returns {any[]}
   */
  getConditionalFields(targetField, fieldsOrGroups) {
    return fieldsOrGroups.map((thing) => {
      if (thing.conditional_logic) {
        const conditions = thing.conditional_logic.map((orLogic) => {
          return orLogic.filter((andLogic) => {
            return andLogic.field === targetField.name
          })
        }).filter(Boolean)

        if (!conditions.length) {
          return false
        }

        return thing
      }

      return false
    })
      .flat()
      .filter(Boolean)
  }


  runConditionalLogic(field) {
    this.getConditionalFields(field, this.fields)
      .forEach((conditionalField) => {
        const matchedFieldEl = this.getFieldEl(
          this.form[conditionalField.name]
        )
        const matchedFieldParentEl = matchedFieldEl.parentElement

        this.controlVisibility(conditionalField, matchedFieldEl)

        if (
          ['radio', 'select', 'checkbox'].indexOf(conditionalField.type) >= 0
        ) {
          conditionalField.options.map((option) => {
            if (!option.conditional_logic) {
              return
            }

            const optionEl = matchedFieldParentEl.querySelector(`${
              conditionalField.type === 'radio' ? 'input' : 'option'
            }[value="${option.label}"]`)

            this.controlVisibility(option, optionEl)
          })
        }
      })

    this.getConditionalFields(field, this.groups)
      .forEach((conditionalGroup) => {
        const matchedGroupEl = this.getFieldEl(
          this.form[conditionalGroup.name]
        )

        this.controlVisibility(conditionalGroup, matchedGroupEl)
      })
  }
}
