<?php
namespace Raptor\Forms;

use Raptor\Forms\Form;
use Raptor\Mail_Monitor;
/**
 * Error response codes:
 * 
 * 3: Required fields were missing.
 * 4: wp_mail returned false.
 * 5: File upload failed validation checks.
 * 6: File size too large.
 * 7: File type not permitted.
 * 9: Forced error for testing via error@example.com
 * 10: reCAPTCHA
 */
class Handler {
    /** @var Form $form */
    var $form = '';

    /** @var int|string $form_id The form ID */
    var $form_id = 0;

    /** @var array $fields The form fields */
    var $fields = [];

    /** @var string $source The source identifier of the form submission */
    var $source = '';

    /** @var null Store the phpmailerException if there is one */
    var $error = null;

    /** @var array Store filepaths to be included in send */
    var $files = [];

    /** @var int $data_id The assigned ID of the data post */
    var $data_id = 0;

    /** @var array $values The form submission data */
    protected $values = [];

    /** @var string The current secret dir for uploads */
    var $secret_uploads_dir;

    /** @var array $graphql_input_values Store the input values from the GraphQL mutution */
    var $graphql_input_values = [];

    /** @var array $graphql_input_files Store the input files from the GraphQL mutution */
    var $graphql_input_files = [];

    /** @var string $test_mode Configure the handler for test mode */
    var $test_mode = '';

    /** @var array $settings The global form settings */
    var $settings = [];

    /** @var array $response The response to send back to the client */
    var $response = [];

    /** @var array $recaptcha_response Store the reCAPTCHA API response */
    var $recaptcha_response = [];

    /**  
     *  Set-up the class ready for handling the request
     */
    function __construct() {
        add_action( 'wp_ajax_raptor_form_resend', [ $this, 'resend' ], 10, 2 );
        add_action( 'wp_ajax_raptor_form_handler', [ $this, 'ajax_request' ], 10, 2 );
        add_action( 'wp_ajax_nopriv_raptor_form_handler', [ $this, 'ajax_request' ], 10, 2 );
        add_action( 'graphql_register_types', [ $this, 'graphql_mutation' ] );
        add_action( 'raptor_delete_expired_form_data', [ $this, 'delete_expired_data' ] );

        if ( !wp_next_scheduled( 'raptor_delete_expired_form_data' ) ) {
            wp_schedule_event( strtotime( '01:00' ), 'daily', 'raptor_delete_expired_form_data' );
        }

        if ( $this->is_ajax_request() || $this->is_graphql_request() ) {
            add_action( 'wp_mail_failed', [ $this, 'failed' ] );
        }
    }


    function graphql_mutation() {
        register_graphql_mutation(
            'raptorFormSubmission',
            [
                'inputFields' => [
                    'formId' => [
                        'type' => 'Int',
                        'description' => 'The form ID',
                    ],
                    'values' => [
                        'type' => 'String',
                        'description' => 'The serialized data from the form'
                    ],
                    'files' => [
                        'type' => 'Upload',
                        'description' => 'Any files to be sent'
                    ]
                ],
                'outputFields' => [
                    'success' => [
                        'type' => 'Boolean',
                        'description' => 'Did mail send successfully.',
                        'resolve' => function( $payload, $args, $context, $info ) {
                            return $payload['success'];
                        }
                    ],
                    'capture' => [
                        'type' => 'Boolean',
                        'description' => 'Did form values capture successfully.',
                        'resolve' => function( $payload, $args, $context, $info ) {
                            return $payload['capture'];
                        }
                    ],
                    'send_mail' => [
                        'type' => 'Boolean',
                        'description' => 'Did mail send successfully.',
                        'resolve' => function( $payload, $args, $context, $info ) {
                            return $payload['send_mail'];
                        }
                    ],
                    'test_mode' => [
                        'type' => 'String',
                        'description' => 'The handler test mode if set.',
                        'resolve' => function( $payload, $args, $context, $info ) {
                            return $payload['test_mode'];
                        }
                    ],
                    'message' => [
                        'type' => 'String',
                        'description' => 'The message to display to the user.',
                        'resolve' => function( $payload, $args, $context, $info ) {
                            return $payload['message'];
                        }
                    ]
                ],
                'mutateAndGetPayload' => function( $input, $context, $info ) {
                    $this->form_id = absint( $input['formId'] );
                    $this->source = $input['source'];
                    $this->graphql_input_values = json_decode( $input['values'], true );
                    $this->graphql_input_files = $input['files'];

                    $this->run();

                    return $this->response;
                }
            ]
        );
    }


    function ajax_request() {
        $this->form_id = isset( $_POST['raptor_form_id'] ) ? $_POST['raptor_form_id'] : 0;
        $this->source = isset( $_POST['raptor_form_source'] ) ? $_POST['raptor_form_source'] : false;

        $this->run();
        $this->send_response();
    }


    /**  
     *  Run the form handler.
     * 
     *  @param int $form_id The ID of the form
     *  @param int $form_data_id The ID of the form submission data
     */
    function run() {
        $this->define_properties();
        $this->get_values();

        $this->response = [
            'success' => false,
            'capture' => false,
            'send_mail' => false,
            'test_mode' => '',
            'message' => ''
        ];

        /**
         * Check if test mode should be enabled.
         * 
         * @param Handler
         */
        $test_modes = apply_filters(
            'raptor_form_handler_test_modes',
            [
                'basic', // Override the email recepient to the input email address
                'mail-failed', // Simulate wp_mail returning false
                'test-tracking' // Test Google Tag Manager event firing, but don't capture or sent mail
            ],
            $this
        );

        $email_field_value = $this->get_field_value( $this->get_email_field_for_send() );

        if ( $email_field_value && preg_match( '/([a-z0-9-]+)@boxchilli\.com$/', $email_field_value, $matches ) ) {
            $test_mode = $matches[1];

            if ( !in_array( $test_mode, $test_modes ) ) {
                $test_mode = 'basic';
            }

            $this->test_mode = $test_mode;
            $this->response['test_mode'] = $test_mode;
        }

        $this->recaptcha_verify();
        $this->validate_required_fields();
        $this->custom_filtering();
        $this->maybe_upload_attachments();

        if (
            $this->settings['submissions']['capture'] !== 'never' &&
            (bool) $this->form->settings['data_capture'] &&
            !$this->is_test_mode( 'test-tracking' )
        ) {
            $this->capture();
        }

        if ( (bool) $this->form->settings['mail_send'] && !$this->is_test_mode( 'test-tracking' ) ) {
            $this->send_mail();
        }

        /**
         * Because `test-tracking` mode doesn't run send_mail(), we must manually set `success` to `true`
         */
        if ( $this->is_test_mode( 'test-tracking' ) ) {
            $this->response['success'] = true;
            $this->response['message'] = 'Thank you for your submission';
        }

        /**
         * After all actions have ran, prepare the response
         */
        if ( (bool) $this->form->settings['mail_send'] ) {
            if ( $this->response['send_mail'] ) {
                $this->response['success'] = true;
                $this->response['message'] = 'Thank you for your submission';

            } else {
                $this->response['message'] = 'There was an error while trying to send your message.';
                $this->response['error'] = [
                    'code' => 'wp_mail_failed',
                    'message' => 'The `wp_mail()` function returned false'
                ];
            }
        }
    }


    /**
     * Resend mail of a captured form submission
     */
    function resend() {
        $data_id = isset( $_REQUEST['form_data_id'] ) ? absint( $_REQUEST['form_data_id'] ) : 0;

        if ( $data_id > 0 ) {
            $this->data_id = $data_id;
            $this->form_id = get_post_meta( $data_id, 'form_id', true );

            $this->define_properties();
            $this->get_values();
            $mail_response = $this->send_mail();

            wp_send_json( $mail_response );
        }

        wp_send_json_error([
            'message' => ""
        ]);
    }


    function define_properties() {
        $this->settings = raptor_get_settings( 'forms' );

        $this->form = raptor_get_form_instance( $this->form_id );

        if ( $this->data_id > 0 ) {
            $this->source = get_post_meta( $this->data_id, 'source', true );
        }
    }


    /**
     * Setup the $values variable with all field values.
     * 
     * @return array
     */
    function get_values() {
        if ( $this->data_id ) {
            $this->values = raptor_get_form_data( $this->data_id );

            return $this->values;
        }

        foreach ( $this->form->fields as $field ) {
            if ( $this->is_ajax_request() ) {
                $this->values[ $field['name'] ] = isset( $_POST[ $field['name'] ] ) ? $_POST[ $field['name'] ] : false;

            } else if ( $this->is_graphql_request() ) {
                $input = $this->graphql_input_values;
                $this->values[ $field['name'] ] = isset( $input[ $field['name'] ] ) ? $input[ $field['name'] ] : false;
            }
        }

        return $this->values;
    }


    /**
     * Send the response back.
     * 
     * @param array $response
     */
    function send_response( array $response = [] ) {
        /**
         * Modify the response before it is sent back to the client.
         * 
         * @param array $response
         * @param Handler $handler
         */
        $response = apply_filters( 'raptor_form_handler_response', wp_parse_args( $response, $this->response ), $this );

        wp_send_json( $response );
    }


    /**
     * Google reCAPTCHA v3 - site verify
     * 
     * @see https://developers.google.com/recaptcha/docs/verify
     */
    function recaptcha_verify() {
        $token = isset( $_POST['grecaptcha-token'] ) ? $_POST['grecaptcha-token'] : false;
        
        if ( !$this->form->recaptcha_enabled() ) {
            return;
        }

        $res = wp_remote_post( 'https://www.google.com/recaptcha/api/siteverify', [
            'body' => [
                'secret' => $this->settings['recaptcha']['secret_key'],
                'response' => $token
            ]
        ]);
        /**
         * If a WP_Error was thrown, it's safer to allow the submission to continue.
         */
        if ( !is_wp_error( $res ) ) {
            $body = json_decode( wp_remote_retrieve_body( $res ), true );
            $this->recaptcha_response = $body;
    
            if ( !$body['success'] ) {
                /**
                 * For unsuccessful verifications, we still want the option to capture.
                 */
                if ( $this->settings['submissions']['capture'] === 'always' ) {
                    $this->capture();
                }

                $this->send_response([
                    'message' => 'Sorry, Google reCAPTCHA thinks you are a robot! If you are not a robot please try again or give us a call!',
                    'error' => [
                        'code' => 'recaptcha_verify_failed',
                        'message' => $body['error-codes'][0]
                    ]
                ]);
            }
        }
    }


    /**  
     *  Validate required fields are present
     */
    function validate_required_fields() {
        // Check all required fields have a value
        $empty_required_fields = [];

        foreach ( $this->form->fields as $field ) {
            if ( $field['type'] == 'submit' ) {
                continue;
            }

            if ( $field['required'] ) {
                if ( $field['type'] == 'file' ) {
                    if ( empty( $this->get_file( $field['name'] ) ) ) {

                        $empty_required_fields[] = $field['name'];
                    }

                } else if ( !$this->get_field_value( $field ) ) {
                    $empty_required_fields[] = $field['name'];
                }
            }
        }

        if ( !empty( $empty_required_fields ) ) {
            $plural = ( count( $empty_required_fields ) == 1 ) ? '' : 's';

            $this->send_response([
                'message' => strval( count( $empty_required_fields ) ) . " field{$plural} " . ( $plural ? 'are' : 'is' ) . ' missing.',
                'error' => [
                    'code' => 'missing_required_fields',
                    'message' => implode( ', ', $empty_required_fields )
                ]
            ]);
        }
    }


    /**
     * Run custom checks to filter 
     */
    function custom_filtering() {
        $filtering_reason = false;
        $filtered_contents = '';

        /**
         * Array of email addresses to check.
         * 
         * @param array $email_addresses
         * @param Handler $handler
         */
        $blacklist_email_addresses = apply_filters( 'raptor/forms/handler/custom_filtering/email_addresses', [], $this );

        /**
         * Array of keywords/phrases to check.
         * 
         * @param array $keywords
         * @param Handler $handler
         */
        $blacklist_keywords = apply_filters( 'raptor/forms/handler/custom_filtering/keywords', [], $this );

        foreach ( $this->form->fields as $field ) {
            $value = $this->get_field_value( $field );

            if ( $field['type'] === 'email' && in_array( $value, $blacklist_email_addresses ) ) {
                $filtering_reason = 'email_address';
                $filtered_contents = $value;
                break;
            }

            if ( !empty( $blacklist_keywords ) && preg_match( '/(' . implode( '|', $blacklist_keywords ) . ')/im', $value, $matches ) ) {
                $filtering_reason = 'keyword';
                $filtered_contents = $matches[0];
                break;
            }
        }

        if ( $filtering_reason ) {
            $this->send_response([
                'message' => "Your enquiry was blocked because it contained <strong>$filtered_contents</strong>",
                'erorr' => [
                    'code' => 'custom_filtering',
                    'message' => $filtered_contents
                ]
            ]);
        }
    }


    /**  
     *  Save the form values to the database
     */
    function capture() {
        do_action( 'raptor_form_handler_capture_before', $this->form );

        $title = date( 'd-m-y-H:i:s' );
        $values = [];

        // Ignores the 'include_in_send' field setting
        foreach ( $this->form->fields as $field ) {
            $_value = $this->get_field_value( $field );

            // Don't capture accept terms and conditions field
            if ( $field['name'] == 'accept_terms' ) {
                continue;
            }

            if ( $field['type'] == 'submit' ) {
                continue;
            }

            // Sanitize field value based on field type
            switch ( $field['type'] ) {
                case 'email':
                    $value = sanitize_email( $_value );
                    break;

                case 'textarea':
                    $value = sanitize_textarea_field( $_value );
                    break;

                case 'select':
                    if ( is_array( $_value ) ) {
                        $value = sanitize_text_field( implode( ', ', $_value ) );
                    } else {
                        $value = sanitize_text_field( $_value );
                    }
                    break;

                case 'checkbox':
                    if ( is_array( $_value ) ) {
                        $value = sanitize_text_field( implode( ', ', $_value ) );
                    } else {
                        $value = sanitize_text_field( $_value );
                    }
                    break;

                default:
                    $value = sanitize_text_field( $_value );
            }

            if ( $field['type'] == 'tel' ) {
                $value = $this->format_phone_number( $value );
            }

            $values[ $field['name'] ] = $value;
        }

        // List of args for the post
        $args = [
            'post_author' => 1,
            'post_title' => $title,
            'post_type' => 'raptor_form_data'
        ];

        $args = apply_filters( 'raptor_form_handler_capture_post_args', $args, $this->form );

        // Create the post and assign post ID
        $this->data_id = wp_insert_post( $args );

        add_post_meta( $this->data_id, 'form_id', $this->form_id );
        add_post_meta( $this->data_id, 'source', $this->source );

        if ( $this->recaptcha_response ) {
            add_post_meta( $this->data_id, 'recaptcha_response', $this->recaptcha_response );
        }
        
        foreach ( $values as $data_key => $data_value ) {
            add_post_meta( $this->data_id, $data_key, $data_value );
        }

        $this->response['capture'] = is_int( $this->data_id );

        do_action( 'raptor_form_handler_capture_after', $this->form, $values, $this->data_id );
    }


    /**  
     *  Send the email
     *
     *  @since 1.0
     */
    function send_mail() {
        $success = wp_mail(
            $this->get_form_recepients(),
            $this->get_form_subject(),
            $this->mail_message_html(),
            $this->get_form_headers(),
            $this->files
        );

        if ( $this->is_test_mode( 'mail-failed' ) ) {
            $success = false;
        }

        $this->response['send_mail'] = $success;
        $this->response['redirect'] = $this->form->settings['redirect'];

        do_action( 'raptor_form_handle_after', $this );

        return $this->response;
    }


    /**  
     *  Create the HTML structure and populate data from the contact form
     *
     *  @since 1.0
     *
     *  @return string The HTML of the email
     */
    function mail_message_html() {
        // Body open
        $html_open = '<html><body>';
        $body_open = '<h1>New Message</h1>';

        $source_title = '';
        $source_link = '';

        list( $object_type, $object_id ) = explode( '-', $this->source );

        if ( $object_type == 'post' ) {
            $source_object = get_post( $object_id );
            $source_title = $source_object->post_title;
            $source_link = get_permalink( $object_id );

        } else if ( $object_type == 'term' ) {
            $source_object = get_term( $object_id );
            $source_title = $source_object->name;
            $source_link = get_permalink( $object_id );
        }

        $body_open = sprintf( '<p>Submitted using the <strong>%s</strong> form via <a href="%s">%s</a>.</p>', $this->form->title, $source_link, $source_title );

        $body_open = apply_filters( 'raptor_form_handler_html_body_open', $body_open, $this->form );

        $table_open = '<table style="width:100%;"><tbody>';

        
        // The form fields
        $table_contents = '';

        $i = 1;
        foreach ( $this->form->fields as $field ) {
            // Honor the include_in_send setting
            if ( isset( $field['include_in_send'] ) && !$field['include_in_send'] ) {
                continue;
            }

            if ( in_array( $field['type'], [ 'file', 'submit' ] ) ) {
                continue;
            }

            $value = $this->get_field_value( $field );

            if ( $field['type'] == 'tel' ) {
                $value = $this->format_phone_number( $value );
            }

            if ( $field['type'] == 'checkbox' ) {
                $value = $value == 'on' ? 'Yes' : 'No';
            }

            $table_contents .= ( $i % 2 ) ? '<tr style="background-color:rgba(120,120,120,0.25);">' : '<tr>';

            $table_contents .= '<td style="width:160px;padding:15px 20px;color:#555555;font-weight:700;text-align:right;">'. $field['label'] .'</td>';
            $table_contents .=  '<td style="padding:15px 20px;">'. wp_unslash( $value ) .'</td>';
            
            $table_contents .= '</tr>';

            $i++;
        }

        // Body close
        $table_close = '</tbody></table>';
        $html_close = '</body></html>';

        // Bring it all together
        $full_body = $html_open . $body_open . $table_open . $table_contents . $table_close . $html_close;

        // Apply any filters to the body HTML and return
        return apply_filters( 'raptor_form_handler_html_full_body', $full_body, $this->form );
    }


    /**
     * Get the recepient email addresses to send to
     * 
     * @return string
     */
    function get_form_recepients() {
        $recepients = $this->form->settings['recepients'];

        if ( empty( $recepients ) ) {
            $recepients = $this->settings['mail']['recipients'];
        }

        $recepients = $this->format_multiple_emails( $recepients );

        if ( $this->is_test_mode( 'basic' ) ) {
            $recepients = $this->get_field_value( $this->get_email_field_for_send() );
        }

        return apply_filters( 'raptor_form_handler_recepients', $recepients, $this->form );
    }


    /**
     * Get the mail subject
     * 
     * @return string
     */
    function get_form_subject() {
        $subject = $this->form->settings['subject'];

        if ( empty( $subject ) ) {
            $subject = $this->settings['mail']['subject'];
        }

        return apply_filters( 'raptor_form_handler_subject', $subject, $this->form );
    }


    /**
     * Get the mail headers
     * 
     * @return string
     */
    function get_form_headers() {
        $headers = [
            'Content-type: text/html; charset=UTF-8'
        ];

        if ( !empty( $this->settings['mail']['cc'] ) ) {
            $headers[] = 'Cc: ' . $this->format_multiple_emails( $this->settings['mail']['cc'] );
        }

        if ( !empty( $this->settings['mail']['bcc'] ) ) {
            $headers[] = 'bcc: ' . $this->format_multiple_emails( $this->settings['mail']['bcc'] );
        }

        $reply_to_value = $this->get_field_value( [ 'name' => $this->form->settings['reply_to'] ] );

        if ( $reply_to_value ) {
            $headers[] = "Reply-To: $reply_to_value";
        }

        return apply_filters( 'raptor_form_handler_headers', $headers, $this->form );
    }


    /**  
     *  Upload attachments if any
     *
     *  @since 1.0
     */
    function maybe_upload_attachments() {
        $files = [];

        foreach ( $this->form->fields as $field ) {
            if ( $field['type'] == 'file' ) {

                if ( $this->validate_attachment( $field ) ) {
                    $file = $this->upload_attachment( $field['name'] );

                    $files[] = $file['path'];

                } else if ( $field['required'] ) {
                    $this->send_response([
                        'message' => 'A file you uploaded has failed security checks.',
                        'error' => [
                            'code' => 'file_validation_failed',
                            'message' => $field['name']
                        ]
                    ]);
                }
            }
        }

        $this->files = $files;
    }


    /**  
     * Find all files and attach to email
     *
     * @since 1.0
     * @param string $field_name
     * @return array
     */
    function upload_attachment( string $field_name = '', $field = null ) {
        if ( $this->is_ajax_request() ) {
            $file = $_FILES[ $field_name ];

        } else if ( $this->is_ajax_request() ) {
            $file = $this->graphql_input_files[ $field_name ];
        }

        $secret_dir = $this->generate_secret_dir( $field_name );
        $this->secret_uploads_dir = $secret_dir;

        add_filter( 'upload_dir', [ $this, 'uploads_filter' ] );
        
        $upload = wp_handle_upload( $file, [ 'test_form' => false ] );

        remove_filter( 'upload_dir', [ $this, 'uploads_filter' ] );

        if ( $upload && !isset( $upload['error'] ) ) {
            add_action( 'raptor_form_handle_after', [ $this, 'delete_uploaded_files' ] );

            return [
                'path' => $upload['file']
            ];
        }

        return [];
    }


    /**
     * Validate the file
     * 
     * @since 1.3.5
     * @param array $field
     * @return bool
     */
    function validate_attachment( array $field ) {
        if ( $this->is_ajax_request() ) {
            $file = $_FILES[ $field['name'] ];

        } else if ( $this->is_ajax_request() ) {
            $file = $this->graphql_input_files[ $field['name'] ];
        }

        $max_file_size = $field['max_file_size'] * 1000000; // Convert MB to bytes

        /**
         * Validate file size.
         */
        if ( $file['size'] > $max_file_size ) {
            $this->send_response([
                'message' => 'A file you uploaded is too large, please try again.',
                'code' => [
                    'code' => 'file_size_too_large',
                    'message' => 'File size is ' . $file['size'] . ', maxmimum allowed is ' . $max_file_size
                ]
            ]);

            return false;
        }

        if ( !empty( $file['error'] ) && $file['error'] !== UPLOAD_ERR_NO_FILE ) {
            return false;
        }

        if ( empty( $file['tmp_name'] ) ) {
            return false;
        }
        
        /** Validate file type */
        $file_type_pattern = raptor_form_get_file_types( $field );
        $file_type_pattern = '/(' . $file_type_pattern . ')$/i';

        return true;
    }


    /**
     * @param array $args
     */
    function uploads_filter( array $args ) {
        $secret_dir = '/' . $this->secret_uploads_dir;

        if ( $secret_dir ) {
            $args['path'] = str_replace( $args['subdir'], '', $args['path'] );
            $args['url'] = str_replace( $args['subdir'], '', $args['url'] );
            $args['subdir'] = $secret_dir;
            $args['path'] .= $secret_dir;
            $args['url'] .= $secret_dir;
    
            if ( !file_exists( $args['path'] ) ) {
                wp_mkdir_p( $args['path'] );
            }
        }

        return $args;
    }


    /**
     * @param string $field_name
     */
    function generate_secret_dir( string $field_name ) {
        return md5( $this->form_id . $field_name );
    }


    /**
     * Delete posts.
     */
    function delete_uploaded_files() {
        $uploads = wp_get_upload_dir();

        foreach ( $this->form->fields as $field ) {
            if ( $field['type'] == 'file' ) {
                $dir = $uploads['basedir'] . '/' . $this->generate_secret_dir( $field['name'] );

                if ( is_dir( $dir ) ) {
                    array_map( 'unlink', glob( $dir . '/*' ) );
                    rmdir( $dir );
                }
            }
        }
    }


    /**  
     *  Used in the Form Handler to format multiple email addresses
     *
     *  @since 1.0
     *
     *  @param string $emails The email addresses as a single string
     *  @return string The email addresses formatted with seperating commas
     */
    function format_multiple_emails( string $emails ) {
        // Remove lines and any spaces
        $emails = trim( preg_replace( '/\s\s+/', ',', $emails ) );

        return $emails;
    }
    

    /**  
     *  Used to chnage the phone number into a more readable format
     *
     *  @since 1.0
     *
     *  @param string $number The raw phone number
     *  @return string
     */
    function format_phone_number( string $number ) {
        $formatted = $number;

        // Format mobile number
        if ( preg_match( '/^07/', $number ) ) {

            $formatted = substr( $number, 0, 3 );
            $formatted .= ' ' . substr( $number , 3, 3 );
            $formatted .= ' ' . substr( $number , 6, 3 );
            $formatted .= ' ' . substr( $number , 9, 2 );
        }

        return $formatted;
    }


    /**  
     * Handle a phpmailerException 
     *
     * @since 1.0
     * @param WP_Error $wp_error
     */
    function failed( \WP_Error $wp_error ) {
        $this->error = $wp_error;

        // Add error object to db
        add_post_meta( $this->data_id, 'raptor_form_mail_error', base64_encode( maybe_serialize( $this->error ) ) );

        Mail_Monitor::notify([
            'form_title' => $this->form->title,
            'errors' => $wp_error->errors
        ]);
    }


    /**  
     *  Delete data posts that exceed a defined expiration time period
     *
     *  @since 1.0
     */
    function delete_expired_data() {
        $days = apply_filters( 'raptor/forms/submissions/expiration_days', 30 );

        if ( $days < 1 ) {
            return;
        }

        $expiration_date = date( 'Y-m-d H:i:s', strtotime( "-$days days", strtotime( date( 'Y-m-d H:i:s' ) ) ) );

        $posts = get_posts([
            'post_type'     => 'raptor_form_data',
            'post_status'   => 'draft',
            'numberposts'   => -1
        ]);

        foreach ( $posts as $post ) {
            setup_postdata( $post );

            // Delete post if post date is after expiration date
            if ( $expiration_date >= $post->post_date ) {
                wp_delete_post( $post->ID );
            }
        }

        wp_reset_postdata();
    }


    /**  
     *  Return a $_FILES array
     *
     *  @since 1.0
     */
    function get_file( $field = '' ) {
        return $_FILES[ $field ];
    }


    /**
     * Get a field value from $_POST.
     * 
     * @param array $field
     * @return string|array|false The POST value or false.
     */
    function get_field_value( array $field ) {
        if ( empty( $field ) || !isset( $this->values[ $field['name'] ] ) ) {
            return false;
        }

        return $this->values[ $field['name'] ];
    }


    /**
     * Get the email field being for Reply-To header.
     */
    function get_email_field_for_send() {
        $email_field = [];

        foreach ( $this->form->fields as $field ) {
            if ( $field['type'] == 'email' ) {
                $email_field = $field;
                break;
            }
        }

        return $email_field;
    }


    /**
     * @return bool
     */
    function is_ajax_request() {
        return defined( 'DOING_AJAX' ) && isset( $_POST['raptor_form_id'] );
    }


    /**
     * @return bool
     */
    function is_graphql_request() {
        return function_exists( 'is_graphql_request' ) && is_graphql_request();
    }


    /**
     * Get the test mode
     * 
     * @return bool
     */
    function is_test_mode( string $mode = '' ) {
        if ( empty( $mode ) ) {
            return (bool) $this->test_mode;
        }

        return $mode === $this->test_mode;
    }
}
