<?php
/**
 * Selective Image Crop
 * 
 * This tool provides the ability to select part of an image as the focus point,
 * ensuring it is not cropped on the frontend. The position is saved as meta data
 * and retrieved during render to add a class to the <img /> that sets the 
 * corresponding object-position value for that selected position.
 */
class Raptor_Selective_Image_Crop {
    /**
     * Add the filters & hooks.
     */
    function __construct() {
        add_filter( 'attachment_fields_to_edit', [ $this, 'attachment_field' ], 10, 2 );
        add_filter( 'attachment_fields_to_save', [ $this, 'save' ], 10, 2 );
        add_filter( 'wp_get_attachment_image_attributes', [ $this, 'add_crop_class' ], 10, 3 );
        add_action( 'graphql_register_types', [ $this, 'register_graphql_fields' ] );
    }

    /**
     * Add the user interface to the attachment fields.
     * 
     * @param array $form_fields
     * @param WP_Post $post
     */
    function attachment_field( $form_fields, $post ) {
        if ( !(bool) preg_match( '/^image\//', get_post_mime_type( $post ) ) ) {
            return $form_fields;
        }

        $image_url = wp_get_attachment_image_url( $post->ID, 'large' );
        $position = get_post_meta( $post->ID, 'selective_image_crop', true );

        $positions = [
            'top_left' => 'Top Left',
            'top_center' => 'Top Center',
            'top_right' => 'Top Right',
            'center_left' => 'Center Left',
            'center_center' => 'Center Center',
            'center_right' => 'Center Right',
            'bottom_left' => 'Bottom Left',
            'bottom_center' => 'Bottom Center',
            'bottom_right' => 'Bottom Right',
        ];
        
        ob_start();
        ?>
        <div class="raptor raptor-selective-image-crop">
            <?php
            echo wp_get_attachment_image( $post->ID, 'large' );
            ?>
            <div class="target-areas">
                <?php
                foreach ( $positions as $key => $label ) {
                    echo '<div>';
                    printf( '<input type="radio" name="raptor[selective_image_crop]" id="%s" value="%s" aria-label="%s" %s>', $key, $key, $label, checked( $position, $key, false ) );
                    printf( '<label for="%s" aria-hidden="true"></label>', $key );
                    echo '</div>';
                }
                ?>
            </div>
        </div>
        <p>Select the area of the image to always remain visible.</p>
        <?php
    
        $html = ob_get_contents();
    
        ob_end_clean();
    
        $form_fields['raptor-selective-image-crop'] = [
            'label' => 'Selective Image Crop',
            'input' => 'html',
            'html'  => $html
        ];
    
        return $form_fields;
    }


    /**
     * Save the position of the crop as meta data to the attachment.
     * 
     * @param array $post
     * @param array $attachment
     */
    function save( $post, $attachment ) {
        $raptor = $_POST['raptor'];
        $selective_image_crop_position = $raptor['selective_image_crop'];

        update_post_meta( $post['ID'], 'selective_image_crop', $selective_image_crop_position );

        return $post;
    }


    /**
     * Add the Selective Image Crop position class to an image attachment.
     */
    function add_crop_class( $attr, $attachment, $size ) {
        $selective_image_crop_position = get_post_meta( $attachment->ID, 'selective_image_crop', true );

        if ( $selective_image_crop_position ) {
            if ( isset( $attr['class'] ) ) {
                $attr['class'] .= ' crop-' . $selective_image_crop_position;
            }
        }

        return $attr;
    }


    /**
     * Expose the crop position to the GraphQL schema.
     */
    function register_graphql_fields() {
        register_graphql_field( 'MediaItem', 'cropPosition', [
            'type' => 'string',
            'resolve' => function( $post ) {
                $position = get_post_meta( $post->ID, 'selective_image_crop', true );

                if ( !$position ) {
                    return null;
                }
                
                return $position;
            }
        ]);
    }
}

new Raptor_Selective_Image_Crop;
