<?php
/**
 * Dynamically load WordPess registered scripts. Dynamic Scripts makes it possible to 
 * lazy load scripts only when they're really needed on the page. This is done using
 * the IntersectionObserver API.
 * 
 * Flexi Blocks can list script dependancies, then Dynamic Scripts will configure the 
 * JavaScript to lazy load the script when the user gets near to the block on the page.
 * 
 * If you dont' want to make use of Dynamic Scripts, for instance you are using Glide.js
 * in a hero, enqueue the script normally with `wp_enqueue_script()`.
 */
class Raptor_Dynamic_Scripts {
    /** @var bool Controls whether or not the JS lazy load function has been loaded. */
    var $has_loaded_js_function = false;

    /**
     * Set the hooks.
     */
    function __construct() {
        add_action( 'wp_footer', [ $this, 'init' ] );
        add_action( 'wp_footer', [ $this, 'google_maps_init' ] );
    }


    /**
     * Build a list of the script dependancies and create the JavaScript scripts.
     */
    function init() {
        global $wp_scripts, $raptor_blocks_library, $flexi_blocks;

        $scripts = [];
        /**
         * Build array of scripts that Flexi Blocks depend on.
         */
        foreach ( $flexi_blocks as $block ) {
            $block_object = $raptor_blocks_library[ $block['acf_fc_layout'] ];

            if ( !$block['settings']['hide'] && $block_object->args['scripts'] ) {
                foreach ( $block_object->args['scripts'] as $script_handle ) {
                    /**
                     * Control if a script dependancy should be allowed.
                     * 
                     * @param bool
                     * @param array $block
                     * @param string $script_handle
                     */
                    if ( apply_filters( 'raptor_dynamic_scripts_block_allow_script', true, $block, $script_handle ) ) {
                        if ( isset( $scripts[ $script_handle ] ) ) {
                            $scripts[ $script_handle ][] = $block_object->name;
        
                        } else {
                            $scripts[ $script_handle ] = [ $block_object->name ];
                        }
                    }
                }
            }
        }

        /**
         * Filter the scripts before they are looped.
         * 
         * @param array $scriptss
         */
        $scripts = apply_filters( 'raptor_dynamic_scripts_scripts', $scripts );

        if ( empty( $scripts ) ) {
            return;
        }

        foreach ( $scripts as $script_handle => $blocks ) {
            $handle = 'raptor-' . $script_handle;
            /**
             * Check the script exists.
             */
            if ( !isset( $wp_scripts->registered[ $handle ] ) ) {
                trigger_error( sprintf( 'The script `%s` is not registered', $handle ), E_USER_WARNING );
                continue;
            }

            /**
             * Skip if the script is already enqueued.
             */
            if ( (bool) $wp_scripts->query( $handle, 'enqueued' ) ) {
                continue;
            }

            $script = $wp_scripts->registered[ 'raptor-' . $script_handle ];
            /**
             * The script src may already have query args set, so use `add_query_arg()` to safely add version.
             */
            $src = add_query_arg(
                [
                    'ver' => $script->ver
                ],
                $script->src
            );

            if ( !$this->has_loaded_js_function ) {
                $this->js_lazy_load_script();
            }

            ?>
            <script id="raptor-<?php echo $script_handle; ?>-lazy-load">
                raptorLazyLoadScript(
                    {
                        src: '<?php echo $src; ?>',
                        handle: '<?php echo $script->handle; ?>'
                    },
                    [
                        <?php
                        foreach ( $blocks as $block_name ) {
                            printf( '...[ ...document.querySelectorAll( \'.flexi-block.block--%s\' ) ],', $block_name );
                        }
                        /**
                         * Allows developers to add custom JavaScript selectors.
                         */
                        echo apply_filters( "raptor_dynamic_scripts_{$script_handle}_observable_elements", '' );
                        ?>
                    ].filter(Boolean)
                );
            </script>
            <?php
        }
    }


    /**
     * Output the JavaScript function used to lazy load a script using
     * the IntersectionObserver API.
     */
    function js_lazy_load_script() {
        $this->has_loaded_js_function = true;

        ?>
        <script id="raptor-lazy-load-function">
            function raptorLazyLoadScript(wpScript, observableElements) {
                if (!observableElements.length) {
                    return;
                }
                let loaded = false
                const observer = new IntersectionObserver(
                    function (entries, observer) {
                        entries.forEach(function(entry) {
                            if (entry.isIntersecting && !loaded) {
                                const script = document.createElement('script');
                                script.type = 'text/javascript';
                                script.src = wpScript.src;
                                script.id = wpScript.handle + '-js';
                                script.setAttribute('data-raptor', 'dynamic-load');
                                document.body.appendChild(script);
                                observer.unobserve(entry.target);
                                loaded = true;

                                console.log(`Dynamically loaded script: ${wpScript.handle}`);
        
                            } else if (loaded) {
                                observer.unobserve(entry.target);
                            }
                        })
                    }, {
                        rootMargin: window.innerHeight + 'px',
                        threshold: 0.1
                    }
                );
                observableElements.forEach(function(el) {
                    observer.observe(el);
                });
            }
        </script>
        <?php
    }


    /**
     * Handles the lazy loading capability of the Google Maps API from maps
     * served via the Google Maps Flexi Block.
     */
    function google_maps_init() {
        /**
         * Ensures this script will only output if the block is present.
         */
        if ( !apply_filters( 'raptor_google_maps_init', has_flexi_block( get_flexi_blocks_with_script( 'google-maps' ) ) ) ) {
            return;
        }

        if ( !defined( 'GOOGLE_API_KEY' ) ) {
            trigger_error( 'Google API key not found', E_USER_WARNING );
        }
    
        ?>
        <script id="raptor-google-maps-init">
            function initRaptorGoogleMaps() {
                if (!window.raptorMaps) {
                    return;
                }
                Object.entries(window.raptorMaps).map(function([ instance, fn ]) {
                    console.log(`loading map: ${instance}`);
                    fn();
                })
            }
        </script>
        <?php
    }
}


new Raptor_Dynamic_Scripts;
