<?php
namespace Raptor\ACF;

use Raptor\ACF\Utils as Utils;
use Raptor\ACF\Field_Types as Field_Types;

/**
 * Register a Flexi Block
 */
class Flexi_Block {

    /** @var string The block name */
    var $name = '';

    /** @var string The block label */
    var $label = '';

    /** @var array The passed settings to the class */
    var $settings = [];

    /** @var array The passed fields to the class */
    var $fields = [];

    /** @var array The passed args to the class */
    var $args = [];

    /** @var array The example data for the block */
    var $example = [];

    /** @var string The description of the the block */
    var $description = '';

    /** @var array Stores the final array used for ACF field data */
    var $acf_field_data = [];

    /** @var array Stores the final array used for global ACF field data */
    var $global_acf_field_data = [];


    /**
     * Prep the Flexi block for init.
     * 
     * @param array $args Arguments for the block
     * @param string $deprecated1 Legacy parameter for the `label`
     * @param array $fields Legacy parameter for the `fields`
     * @param array $dep_args Legacy parameter for the `args`
     */
    function __construct( $args = [], $deprecated1 = '', $deprecated2 = [], $deprecated3 = [] ) {
        if ( !is_array( $args ) ) {
            $name = $args;
            /**
             * @since v1.3 Set `$args` up to use the revised format
             */
            $args = array_merge(
                $deprecated3,
                [
                    'name' => $name,
                    'title' => $deprecated1,
                    'fields' => $deprecated2
                ]
            );
        }

        $this->name = $args['name'];
        $this->label = $args['title'];
        $fields = $args['fields'];

        if ( apply_filters( 'raptor_flexi_block_process_fields', true, $args['name'] ) ) {
            $fields = $this->process_fields( $args['fields'] );
        }

        $this->fields = $fields;
        
        $default_args = [
            'description'   => '',
            'fields'        => [],
            'settings'      => [],
            'example'       => [],
            'block_before'  => false,
            'block_after'   => false,
            'allow_global'  => true,
            'scripts'       => [],
            /**
             * Control the minimum number of instances a block can be used.
             */
            'min'           => '',
            /**
             * Control the maximum number of instances a block can be used.
             */
            'max'           => '',
            /**
             * Full access to override the layout array
             * 
             * @deprecated
             */
            'overrides'     => []
        ];
        $this->args = wp_parse_args( $args, $default_args );

        $default_settings = apply_filters(
            'raptor_flexi_block_default_settings',
            [
                'id'            => true,
                'background'    => [
                    'choices'       => [
                        'white' => 'White',
                        'black' => 'Black',
                        'grey'  => 'Grey'
                    ],
                    'default_value' => 'white'
                ]
            ],
            $this
        );

        $this->example = $this->args['example'];
        $this->description = $this->args['description'];

        if ( $this->args['settings'] !== false && is_array( $this->args['settings'] ) ) {
            $this->args['settings'] = $this->parse_settings( $this->args['settings'], $default_settings );
        }

        // Override $args again to add "Use global block" setting
        if ( $this->args['allow_global'] && Utils\global_block_exists( $args['name'] ) ) {
            if ( !is_array( $this->args['settings'] ) ) {
                $this->args['settings'] = [];
            }

            $this->args['settings'] = [ 'use_global_block' => true ] + $this->args['settings'];
        }

        $this->args['settings'] = [ 'hide_block' => true ] + $this->args['settings'];

        /**
         * Ensure that `scripts` is always an array.
         */
        if ( gettype( $this->args['scripts'] ) == 'string' ) {
            $this->args['scripts'] = [ $this->args['scripts'] ];
        }

        if ( Utils\block_exists( $args['name'] ) ) {
            trigger_error( 'The block "'. $args['title'] .'" has already been registered to the Raptor Blocks Library.', E_USER_NOTICE );
            return;
        }

        $this->init();
    }


    /**
     * Where the bulk of the block data is constructed
     */
    public function init() {
        $default_acf_field_data = [
            'key'           => 'layout_block__' . $this->name,
            'name'          => $this->name,
            'label'         => $this->label,
            'display'       => 'block',
            'sub_fields'    => [],
            'min'           => '',
            'max'           => ''
        ];

        $acf_field_data = wp_parse_args( $this->args['overrides'], $default_acf_field_data );

        $acf_field_data['min'] = $this->args['min'];
        $acf_field_data['max'] = $this->args['max'];

        // GraphQL support.
        if ( raptor_supports( 'headless' ) ) {
            $acf_field_data['show_in_graphql'] = true;
        }

        // Add tab for main block fields, only if fields set.
        if ( !empty( $this->fields ) ) { 
            array_unshift( $this->fields, $this->block_tab( 'Fields' ) );
        }

        if ( $this->args['block_before'] ) {
            array_unshift( $this->fields, $this->block_tab( 'Heading' ), $this->block_before() );
        }

        if ( $this->args['block_after'] ) {
            $this->fields[] = $this->block_tab( 'Additional content' );
            $this->fields[] = $this->block_after();
        }

        if ( $this->args['settings'] ) {
            /**
             * Create settings group field
             */
            $settings_field = $this->setup_settings();
            $settings_field['sub_fields'] = [];
            
            foreach ( $this->args['settings'] as $setting => $value ) {
                /**
                 * Only support pre-defined settings right now
                 */
                if ( $value && method_exists( $this, 'setting_' . $setting ) ) {
                    $settings_field['sub_fields'][] = call_user_func( [ __CLASS__, 'setting_' . $setting ] );

                } else if ( isset( $value['type'] ) ) {
                    /**
                     * Allows custom settings to be added.
                     */
                    $value['key'] = apply_filters( 'raptor_flexi_block_custom_setting_key', $this->name . '_block_custom_setting_' . $value['name'], $setting, $this );
                    $settings_field['sub_fields'][] = $value;
                }
            }

            // Append settings to end of all sub fields of block
            $this->fields[] = $this->block_tab( 'Settings' );
            $this->fields[] = $settings_field;
        }

        
        $acf_field_data['sub_fields'] = $this->setup_conditional_logic( $this->fields );

        $this->acf_field_data = $acf_field_data;


        if ( $this->args['allow_global'] ) {
            $block_settings_index = array_search( $this->name . '_settings', array_column( $acf_field_data['sub_fields'], 'key' ) );

            $use_global_setting_index = array_search(
                $this->name . '_block_setting_use_global',
                array_column( $acf_field_data['sub_fields'][ $block_settings_index ]['sub_fields'], 'key' )
            );

            if ( $use_global_setting_index >= 0 ) {
                unset( $acf_field_data['sub_fields'][ $block_settings_index ]['sub_fields'][ $use_global_setting_index ] );
            }

            $this->global_acf_field_data = $acf_field_data;
        }
        /**
         * Finally, add the block to the library
         */
        $this->add_block_to_library();
    }


    /**
     * Returns the complete ACF field data array
     * 
     * @return array The ACF field array
     */
    public function get_acf_field_data( bool $for_global = false ) {

        if ( $for_global ) {
            if ( $this->args['allow_global'] ) {
                return $this->global_acf_field_data;
            }

            return [];
        }

        return $this->acf_field_data;
    }


    /**
     * Returns an ACF group field for the block settings
     * 
     * @return array The ACF group field for settings
     */
    public function setup_settings( array $args = [] ) {
        $defaults = [
            'key'       => $this->name . '_settings',
            'wrapper'   => [
                'class'     => 'hide-label'
            ]
        ];   
        $args = wp_parse_args( $args, $defaults );
        
        return Field_Types\group_field( 'Settings', $args );
    }


    /**
     * Returns the use global setting field
     * 
     * @return array
     */
    public function setting_use_global_block() {
        $query = new \WP_Query([
            'post_type' => 'flexi_global',
            'meta_key' => 'flexi_block',
            'meta_value' => $this->name
        ]);

        $global_blocks = [];

        foreach ( $query->posts as $post ) {
            $global_blocks[ $post->ID ] = $post->post_title;
        }

        return apply_filters(
            'raptor_acf_flexi_block_setting_use_global',
            Field_Types\select_field(
                'Global Block',
                [
                    'key'       => $this->name . '_block_setting_use_global',
                    'name'      => 'use_global',
                    'choices'   => $global_blocks,
                    'map'       => 'setting_global_block',
                    'wrapper'   => [
                        'width'     => 40
                    ],
                    'allow_null' => true
                ]
            )
        );
    }


    /**
     * Returns the ID setting field
     * 
     * @return array
     */
    public function setting_id() {
        return apply_filters(
            'raptor_acf_flexi_block_setting_id',
            Field_Types\text_field(
                'HTML ID',
                [
                    'key'       => $this->name . '_block_setting_id',
                    'name'      => 'id',
                    'source'    => 'setting_id',
                    'wrapper'   => [
                        'width'     => '33'
                    ]
                ]
            )
        );
    }


    /**
     * Returns the background setting field
     * 
     * @return array
     */
    public function setting_background() {
        return apply_filters(
            'raptor_acf_flexi_block_setting_background',
            Field_Types\select_field(
                'Background',
                [
                    'key'           => $this->name . '_block_setting_background',
                    'name'          => 'background',
                    'choices'       => $this->args['settings']['background']['choices'],
                    'default_value' => $this->args['settings']['background']['default_value'],
                    'source'        => 'setting_background',
                    'wrapper'       => [
                        'width' => '33'
                    ]
                ]
            )
        );
    }


    /**
     * Returns the layout setting field
     * 
     * @return array
     */
    public function setting_layout() {
        return apply_filters(
            'raptor_acf_flexi_block_setting_layout',
            Field_Types\button_group_field(
                'Layout',
                [
                    'key'           => $this->name . '_block_setting_layout',
                    'name'          => 'layout',
                    'choices'       => $this->args['settings']['layout']['choices'],
                    'default_value' => $this->args['settings']['layout']['default_value'],
                    'source'        => 'setting_layout'
                ]
            )
        );
    }


    /**
     * Returns the hide setting field
     * 
     * @return array
     */
    public function setting_hide_block() {
        return apply_filters(
            'raptor_acf_flexi_block_setting_hide',
            Field_Types\true_false_field(
                'Hide Block',
                [
                    'key'       => $this->name . '_block_setting_hide',
                    'name'      => 'hide',
                    'source'    => 'setting_hide',
                    'wrapper'   => [
                        'width'     => '15'
                    ]
                ]
            )
        );
    }


    /**
     * Splits the layout sub fields into parts
     * 
     * @param string $name
     * @return array An ACF tab field
     */
    public function block_tab( string $name ) {
        return Field_Types\tab_field(
            $name,
            [
                'key' => "{$this->name}_block_tab_" . sanitize_title( $name )
            ]
        );
    }


    /**
     * The block before field.
     * 
     * @return array An ACF wysiwyg field
     */
    public function block_before() {
        return Field_Types\wysiwyg_field(
            'Block before',
            [
                'key'       => "{$this->name}_block_block-before",
                'source'    => 'block_before',
                'wrapper'   => [
                    'class'     => 'hide-label mini-editor'
                ]
            ]
        );
    }


    /**
     * The block after field.
     * 
     * @return array An ACF wysiwyg field
     */
    public function block_after() {
        return Field_Types\wysiwyg_field(
            'Block after',
            [
                'key'       => "{$this->name}_block_block-after",
                'source'    => 'block_after',
                'wrapper'   => [
                    'class'     => 'hide-label mini-editor'
                ]
            ]
        );
    }


    /**
     * Adds the block instance to the global Blocks Library
     */
    public function add_block_to_library() {
        global $raptor_blocks_library;

        $raptor_blocks_library[ $this->name ] = $this;
    }


    /**
     * Parses the multidimensional array for the block settings. wp_parse_args for multidimensional arrays
     * 
     * With thanks: https://mekshq.com/recursive-wp-parse-args-wordpress-function/
     * 
     * @param array $defaults
     * @param array $args
     * @return array The parsed array
     */
    private function parse_settings( array $args, array $defaults ) {
        $args = (array) $args;
        $defaults = (array) $defaults;
    
        $result = $defaults;
    
        foreach ( $args as $k => &$v ) {
            if ( is_array( $v ) && isset( $result[ $k ] ) && $k != 'choices' ) {
                $result[ $k ] = $this->parse_settings( $v, $result[ $k ] );
            } else {
                $result[ $k ] = $v;
            }
        }
    
        return $result;
    }


    /**
     * Process the fields, setting keys scoped to the block.
     * 
     * @param array $fields
     * @return array
     */
    private function process_fields( array $fields ) {
        return array_map( function( $field ) {
            $field['key'] = $this->name . '-' . sanitize_title( $field['label'] );

            if ( $field['type'] == 'repeater' ) {
                $field['sub_fields'] = $this->process_fields( $field['sub_fields'] );
            }

            return $field;
        }, $fields );
    }


    /**
     * Add conditional logic to fields when global block is set.
     * 
     * @param array $fields
     * @return array
     */
    private function setup_conditional_logic( array $fields ) {
        if ( !$this->args['allow_global'] && !Utils\global_block_exists( $this->name ) ) {
            return $fields;
        }

        $conditional_logic = [
            [
                [
                    'field' => $this->name . '_block_setting_use_global',
                    'operator' => '==empty'
                ]
            ]
        ];

        $fields = array_map( function( $field ) use( $conditional_logic ) {
            if ( $field['name'] == 'settings' && isset( $field['sub_fields'] ) ) {
                $field['sub_fields'] = array_map( function( $sub_field ) use( $conditional_logic ) {
                    if ( $sub_field['name'] == 'use_global' ) {
                        return $sub_field;
                    }

                    $sub_field['conditional_logic'] = $conditional_logic;

                    return $sub_field;
                }, $field['sub_fields'] );
            }

            // Hide all but the settings tab.
            if ( ( $field['type'] == 'tab' && $field['name'] != 'tab_settings' ) ) {
                $field['conditional_logic'] = $conditional_logic;
            }

            return $field;
        }, $fields );

        return $fields;
    }


    /**
     * Get the URL for the block preview.
     * 
     * @return string
     */
    public function preview_url() {
        $url = '';
        $uploads = wp_upload_dir();
        $basedir = $uploads['basedir'];
        $baseurl = $uploads['baseurl'];

        if ( is_multisite() ) {
            $basedir = WP_CONTENT_DIR . '/uploads';
            $baseurl = WP_CONTENT_URL . '/uploads';
        }

        $path = "/raptor/flexi-block-previews/{$this->name}.jpeg";
        $exists = file_exists( $basedir . $path );

        if ( $exists ) {
            $timestamp = get_option( 'raptor_flexi_block_preview' );
            $url = $baseurl . $path;

            if ( $timestamp ) {
                $url = add_query_arg( [ 'timestamp' => $timestamp ], $url );
            }
        } else {
            $url = raptor_get_asset_url( 'assets/images/flexi-block-preview-not-available.png?version=' . raptor()->version );
        }


        return apply_filters( 'raptor/flexi/block/preview_url', $url, $this );
    }
}
