import Swiper from 'swiper'
import { EffectFade, Pagination } from 'swiper/modules'


export default class FormGroupsWithSwiper {
  /** @type {Swiper | undefined} */
  swiper = null

  /** @type {HTMLElement} */
  swiperEl = null

  /** @type {HTMLFormElement} */
  form = null

  /** @type {HTMLButtonElement | null} */
  prevButton = null

  /** @type {HTMLButtonElement | null} */
  nextButton = null

  constructor(form, swiperEl) {
    this.form = form
    this.swiperEl = swiperEl
  }


  init() {
    this.swiper = new Swiper(this.swiperEl, {
      modules: [EffectFade, Pagination],
      slidesPerView: 1,
      spaceBetween: 32,
      effect: 'fade',
      dynamicBullets: true,
      touchRatio: 0,
      on: {
        slideChange: () => {
          this.validateSlide()
        }
      }
    })

    /**
     * User journey must always start from slide index 0
     */
    this.swiper.raptor = {
      groupJourney: [0]
    }

    this.prevButton = this.form.querySelector('.button-prev')
    this.nextButton = this.form.querySelector('.button-next')

    this.navigation()
  }


  navigation() {
    this.prevButton?.addEventListener('click', (event) => {
      event.preventDefault()

      const prevIndex = 
        this.swiper.raptor?.groupJourney.at(-2) || 
        this.swiper.activeIndex - 1
      this.swiper.slideTo(prevIndex)
      
      if (this.swiper.raptor?.groupJourney) {
        this.swiper.raptor.groupJourney.pop()
      }
    })

    this.nextButton?.addEventListener('click', (event) => {
      event.preventDefault()

      const nextIndex = 
        this.swiper.raptor?.nextGroupIndex || 
        this.swiper.activeIndex + 1
      this.swiper.slideTo(nextIndex)

      if (Array.isArray(this.swiper.raptor?.groupJourney)) {
        this.swiper.raptor.groupJourney.push(nextIndex)
      }
    })
  }


  validateSlide() {
    const { swiper } = this
    /** @type {HTMLElement} */
    const currentSlide = swiper.slides[swiper.activeIndex]
    const fields = currentSlide.querySelector('.fields')
    const requiredInputs = fields.querySelectorAll(
      '[required], [data-required="true"]'
    )

    this.handleLastSlide()

    // Attach event listeners to all input fields for real-time validation
    requiredInputs.forEach((input) => {
      input.addEventListener('input', this.validateFields.bind(this, requiredInputs))
    })

    this.validateFields(requiredInputs)
  }


  /**
   * Check if all fields are valid
   * 
   * @param {NodeListOf<HTMLElement>} requiredInputs
   */
  validateFields(requiredInputs) {
    const allValid = Array.from(requiredInputs)
      .every((input) => input.checkValidity())

    this.disableNextButton(!allValid)
  }


  handleLastSlide() {
    const { nextButton, swiper } = this

    if (nextButton && swiper.activeIndex === swiper.slides.length - 1) {
      nextButton.innerHTML = 'Submit'

      setTimeout(() => {
        nextButton.addEventListener('click', this.submitForm)
      }, 50)
    } else {
      nextButton.innerHTML = 'Next'

      setTimeout(() => {
        nextButton.removeEventListener('click', this.submitForm)
      }, 50)
    }
  }


  /**
   * @param {boolean} disabled 
   */
  disableNextButton(disabled) {
    this.nextButton.disabled = disabled
  }


  submitForm() {
    const formSubmitBtn = this.form.querySelector('button[type="submit"]')
    const allInputs = this.form.querySelectorAll(
      '[required], [data-required="true"]'
    )

    allInputs.forEach(input => {
      input.removeAttribute('required')
      input.removeAttribute('data-required')
    })

    formSubmitBtn.click()
  }
}
