<?php

namespace Raptor;

use Raptor\ACF\Utils;
use Raptor\ACF\Group;
use Raptor\ACF\Field_Types;
use Raptor\Headless\GraphQL_Register_Fields;


class Schema {
    static $types = [];

    function __construct() {
        add_action( 'acf/init', [ $this, 'prepare_types' ], 10 );
        add_action( 'acf/init', [ $this, 'register_acf_options' ], 15 );
        add_action( 'acf/init', [ $this, 'register_acf_post_group' ] );
        add_action( 'wp_head', [ $this, 'output_global_script' ] );
        add_action( 'wp_head', [ $this, 'output_post_script' ] );
    }


    static function prepare_types() {
        $schema_types = [
            [
                'type' => 'local_business',
                'label' => 'Local Business',
                'fields' => [
                    'setup' => [
                        Field_Types\image_field(
                            'Image',
                            [
                                'instructions' => 'This is most likely the company Logo',
                                'return_format' => 'url'
                            ]
                        ),
                        Field_Types\text_field(
                            'Type',
                            [
                                'default_value' => 'LocalBusiness'
                            ]
                        ),
                        Field_Types\text_field(
                            'Name'
                        ),
                        Field_Types\text_field(
                            'URL'
                        ),
                        Field_Types\text_field(
                            'Telephone'
                        ),
                        Field_Types\text_field(
                            'Price Range'
                        ),
                        Field_Types\text_field(
                            'Street'
                        ),
                        Field_Types\text_field(
                            'City'
                        ),
                        Field_Types\text_field(
                            'Post Code'
                        ),
                        Field_Types\text_field(
                            'Country'
                        )
                    ],
                    'settings' => []
                ],
                'callback' => [ __CLASS__, 'local_business' ],
                'group_args' => [
                    'instructions' => 'This is the local_business schema, more details about it can be found <a href="https://developers.google.com/search/docs/appearance/structured-data/local-business" target="_blank">here</a>.'
                ]
            ],
            [
                'type' => 'aggregate',
                'label' => 'Aggregate',
                'fields' => [
                    'setup' => [
                        Field_Types\text_field(
                            'Brand Name'
                        ),
                        Field_Types\text_field(
                            'Name'
                        ),
                        Field_Types\text_field(
                            'Description',
                            [
                                'instructions' => 'A short description of the business.'
                            ]
                        ),
                        Field_Types\text_field(
                            'Star Rating',
                            [
                                'instructions' => 'This is a star rating out of 5'
                            ]
                        ),
                        Field_Types\text_field(
                            'Review Count',
                            [
                                'instructions' => 'How many reviews are there?'
                            ]
                        )
                    ],
                    'settings' => []
                ],
                'callback' => [ __CLASS__, 'aggregate' ],
                'group_args' => [
                    'instructions' => 'This is the aggregate rating schema, more details about it can be found <a href="https://developers.google.com/search/docs/appearance/structured-data/review-snippet" target="_blank">here</a>.'
                ]
            ]
        ];

        /**
         * Allow for schema types to be added.
         * 
         * @param array $schema_types
         */
        $schema_types = apply_filters( 'raptor/schema/types', $schema_types );

        self::$types = $schema_types;

        return $schema_types;
    }


    /**
     * Register the Site Options
     */
    function register_acf_options(): void {
        foreach ( self::prepare_types() as $schema_type ) {
            Utils\add_settings_tab([
                'group' => 'schema',
                'label' => $schema_type['label'],
                'fields' => array_merge(
                    [
                        Field_Types\tab_field( 'Setup' ),
                    ],
                    $schema_type['fields']['setup'],
                    [
                        Field_Types\tab_field( 'Settings' ),
                        Field_Types\true_false_field(
                            'Enabled',
                            [
                                'instructions' => 'Controls if the schema should be loaded at all.'
                            ]
                        ),
                        $this->get_available_post_types()
                    ],
                    $schema_type['fields']['settings']
                ),
                'group_args' => array_merge(
                    [
                        'name' => 'schema_' . $schema_type['type'],
                        'key' => 'schema_' . $schema_type['type']
                    ],
                    $schema_type['group_args']
                )
            ]);
        }

        Utils\register_settings_group([
            'name' => 'schema',
            'title' => 'Schema',
            'menu_title' => 'Schema',
            'redirect' => false,
            'position' => 54,
            'show_in_graphql' => true,
            'parent_slug' => 'raptor-settings'
        ]);
    }


    /**
     * Register the ACF group for a single post
     */
    function register_acf_post_group(): void {
        $post_types = apply_filters( 'raptor/schema/post_types', [ 'page' ] );

        $location = array_map( function( string $post_type ) {
            return [
                [
                    'param' => 'post_type',
                    'operator' => '==',
                    'value' => $post_type
                ]
                ];
        }, $post_types );

        new Group(
            'post_schema',
            [
                'title' => 'Structured Data',
                'fields' => [
                    Field_Types\textarea_field(
                        'JSON',
                        [
                            'name' => 'custom_schema',
                            'key' => 'custom_schema'
                        ]
                    )
                ],
                'location' => $location,
                'menu_order' => 50
            ]
        );
    }


    /**
     * Output the `<script>` tag in the head
     */
    function output_global_script() {
        $page_id = get_queried_object_id();

        foreach ( self::prepare_types() as $schema ) {
            $schema_data = get_field( 'schema_' . $schema['type'], 'options' );
            $pages = $schema_data['pages'];

            if ( !isset( $pages ) ) {
                continue;
            }

            if ( !$schema_data['enabled'] ) {
                continue;
            }

            if ( is_singular() && ( ( is_array( $pages ) && in_array( $page_id, $pages ) ) || !$pages ) ) {
                $json = call_user_func( $schema['callback'], $schema_data );
                /**
                 * Allow the JSON to be modified before encoding into the <script> tag.
                 * 
                 * @param array $json
                 * @param array $schema
                 */
                $json = apply_filters( 'raptor/schema/json', $json, $schema );
                $json = apply_filters( 'raptor/schema/json/' . $schema['type'], $json, $schema );

                printf( '<!-- %s Schema -->', $schema['label'] );
                printf( '<script type="application/ld+json">%s</script>', wp_json_encode( $json ) );
            }
        }
    }


    /**
     * The Local Business schema
     * 
     * @param array $data
     * @return array
     */
    static function local_business( array $data ): array {
        $post_code = isset( $data['post_code'] ) ? $data['post_code'] : $data['post-code'];

        return [
            '@context' => 'https://schema.org',
            '@type' => $data['type'],
            'name' => $data['name'],
            'image' => $data['image'],
            '@id' => '',
            'url' => $data['url'],
            'telephone' => $data['telephone'],
            'address' => [
                '@type' => 'PostalAddress',
                'streetAddress' => $data['street'],
                'addressLocality' => $data['city'],
                'postalCode' => $post_code,
                'addressCountry' => $data['country']
            ],
        ];
    }


    /**
     * The Aggregate Rating schema
     * 
     * @param array $data
     * @return array
     */
    static function aggregate( array $data ): array {
        $brand = isset( $data['brand_name'] ) ? $data['brand_name'] : $data['brand-name'];
        $star_rating = isset( $data['star_rating'] ) ? $data['star_rating'] : $data['star-rating'];
        $review_count = isset( $data['review_count'] ) ? $data['review_count'] : $data['review-count'];

        return [
            '@context' => 'http://www.schema.org',
            '@type' => 'AggregateRating',
            'brand' => $brand,
            'name' => $data['name'],
            'description' => $data['description'],
            'aggregateRating' => [
                '@type' => 'aggregateRating',
                'ratingValue' => $star_rating,
                'ratingCount' => $review_count,
                'bestRating' => '5',
                'worstRating' => '0'
            ]
        ];
    }


    /**
     * Output schema saved against a single post
     */
    function output_post_script() {
        $custom_schema = get_field( 'custom_schema' );

        if ( $custom_schema ) {
            echo $custom_schema;
        }
    }


    /**
     * Load the ACF post object field for posts available to load schema on
     * 
     * @return array
     */
    function get_available_post_types(): array {
        return Field_Types\post_object_field(
            'Pages',
            [
                'instructions' => 'Select the pages you want this to show on, if you leave it blank it will show on all pages.',
                'post_type' => apply_filters( 'raptor/schema/post_types', [ 'page' ] )
            ]
        );
    }    
}

new Schema;


if ( class_exists( 'Raptor\Headless\GraphQL_Register_Fields' ) ) {
    class GraphQL extends GraphQL_Register_Fields {

        function global() {
            register_graphql_object_type( 'Schema', [
                'description' => 'Structured data',
                'fields' => [
                    'type' => [ 'type' => 'String' ],
                    'pages' => [ 'type' => [ 'list_of' => 'Int' ] ],
                    'json' => [ 'type' => 'String' ]
                ]
            ]);
    
            register_graphql_field( 'RootQuery', 'raptorSchemas', [
                'type' => [ 'list_of' => 'Schema' ],
                'resolve' => function() {
                    $schemas = [];
    
                    foreach ( Schema::prepare_types() as $schema ) {
                        if ( $schema['type'] === 'aggregate' ) {
                            continue;
                        }
    
                        $schema_data = get_field( 'schema_' . $schema['type'], 'options' );
            
                        $json = call_user_func( $schema['callback'], $schema_data );
                        /**
                         * Allow the JSON to be modified
                         * 
                         * @param array $json
                         * @param array $schema
                         */
                        $json = apply_filters( 'raptor/schema/json', $json, $schema );
                        $json = apply_filters( 'raptor/schema/json/' . $schema['type'], $json, $schema );
    
                        $schemas[] = [
                            'type' => $schema['type'],
                            'pages' => $schema_data['pages'],
                            'json' => json_encode( $json )
                        ];
                    }
    
                    return !empty( $schemas ) ? $schemas : null;
                }
            ]);
        }
    
    
        function post_types() {
            $post_types = apply_filters( 'raptor/schema/post_types', [ 'page' ] );
    
            foreach ( $post_types as $post_type ) {
                register_graphql_field(
                    $post_type,
                    'raptorSchema',
                    [
                        'type' => 'String',
                        'description' => 'Structured data to add to the page.',
                        'resolve' => function() {
                            return get_field( 'custom_schema' );
                        }
                    ]
                );
            }   
        }
    }
    
    new GraphQL([
        'global',
        'post_types'
    ]);    
}
