<?php
namespace Raptor\Headless;

use Raptor\Cache_Manager;
use Raptor\Headless;

class Nextjs {
    function __construct() {
        add_filter( 'raptor_js_object', [ $this, 'raptor_js_object' ] );
        /**
         * AJAX processing
         */
        add_action( 'wp_ajax_raptor_headless_vercel_get_deployment_status', [ __CLASS__, 'vercel_get_deployment_status' ], 10 );
        add_action( 'wp_ajax_raptor_headless_vercel_set_deployment_status', [ __CLASS__, 'vercel_set_deployment_status' ], 10 );
        add_action( 'cli_init', [ $this, 'cli_init' ] );
    }


    /**
     * Add CLI commands
     */
    function cli_init() {
        \WP_CLI::add_command( 'raptor headless vercel-redeploy', [ $this, 'cli_vercel_redeploy' ] );
    }


    /**
     * Enqueue scripts.
     */
    function enqueue_scripts() {
        wp_register_script( 'raptor-headless', raptor_get_asset_url( 'assets/js/headless.js' ), [], raptor()->version, [ 'in_footer' => true ] );
        wp_enqueue_script( 'raptor-headless' );
    }


    /**
     * Trigger a Vercel redeploy.
     * 
     * @param array $args
     * @param array $assoc_args
     */
    function cli_vercel_redeploy( $args, $assoc_args ) {
        $assoc_defaults = [
            'graphql' => false
        ];

        $assoc_args = wp_parse_args( $assoc_args, $assoc_defaults );
        $graphql = filter_var( $assoc_args['graphql'], FILTER_VALIDATE_BOOLEAN );

        if ( $graphql ) {
            Cache_Manager::graphql_purge_all();
        }

        self::trigger_vercel_deploy_hook();
    }


    /**
     * Filter the Raptor JavaScript object for admin use.
     * 
     * @param array $js_object
     */
    function raptor_js_object( array $js_object ) {
        if ( is_admin() ) {
            $js_object['headless'] = [
                'vercel' => [
                    'teamId' => defined( 'VERCEL_TEAM_ID' ) ? VERCEL_TEAM_ID : false,
                    'projectId' => defined( 'VERCEL_PROJECT_ID' ) ? VERCEL_PROJECT_ID : false,
                    'deployHook' => defined( 'VERCEL_DEPLOY_HOOK' ) ? VERCEL_DEPLOY_HOOK : false,
                    'apiToken' => defined( 'VERCEL_API_TOKEN' ) ? VERCEL_API_TOKEN : false
                ]
            ];
        }

        return $js_object;
    }


    static function revalidate_post( \WP_Post $post ) {
        $tag = str_replace( rtrim( home_url(), '/' ), '', get_permalink( $post->ID ) );

        if ( $post->post_type == 'flexi_global' ) {
            $tag = sprintf( 'flexi-preview_%s_%s', get_post_meta( $post->ID, 'flexi_block', true ), $post->ID );
        }

        return self::revalidate_tag( $tag );
    }


    /**
     * Send revalidate request to Next.js app.
     * 
     * @param string $tag The tag or path
     */
    static function revalidate( string $tag, string $type, string $layout = '' ) {
        $frontend_url = Headless::get_frontend_url();
        $preview_secret = Headless::get_preview_secret();

        if ( !$preview_secret || !$frontend_url ) {
            return false;
        }

        $transient_name = 'revalidate_' . $tag;
        /**
         * Helps throttle how often revalidate can occur.
         */
        if ( get_transient( $transient_name ) ) {
            return false;
        }

        $query_args = [
            'secret' => $preview_secret
        ];

        if ( $type === 'path' ) {
            $query_args['path'] = $tag;

            if ( $layout ) {
                $query_args['layout'] = $layout;
            }
        } else {
            $query_args['tag'] = $tag;
        }

        $url = add_query_arg(
            $query_args,
            $frontend_url . 'api/revalidate/' . ( $type === 'path' ? 'path' : '' )
        );

        try {
            $response = wp_remote_get( $url, [ 'timeout' => 10 ] );

            if ( is_wp_error( $response ) ) {
                return false;
            }
    
            if ( wp_remote_retrieve_response_code( $response ) !== 200 ) {
                return false;
            }
    
            $body = json_decode( wp_remote_retrieve_body( $response ), true );
        
            if ( $body['revalidated'] ) {
                set_transient( $transient_name, true, 10 );
    
                return true;
            }

            return false;
        } catch ( \Exception $e ) {
            return false;
        }
    }


    /**
     * Revalidate a tag.
     * 
     * @param string $tag The tag
     * @return bool Whether the revalidation successed or failed.
     */
    static function revalidate_tag( $tag ) {
        return self::revalidate( $tag, 'tag' );
    }


    /**
     * Revalidate a tag.
     * 
     * @param string $tag The tag
     * @return bool Whether the revalidation successed or failed.
     */
    static function revalidate_path( string $path, string $layout = '' ) {
        return self::revalidate( $path, 'path', $layout );
    }


    /**
     * Revalidate the global data, this is usually shared across
     * all pages on a site.
     */
    static function revalidate_global_data() {
        return self::revalidate_tag( 'global_data' );
    }

    /**
     * Revalidate the global data, this is usually shared across
     * all pages on a site.
     */
    static function revalidate_all_data() {
        return self::revalidate_path( '/', 'layout' );
    }


    /**
     * --------------------------------------
     * 
     * Vercel functions
     * 
     * --------------------------------------
     */

    /**
     * Trigger a Deploy Hook on Vercel.
     */
    static function trigger_vercel_deploy_hook() {
        if ( !defined( 'VERCEL_DEPLOY_HOOK' ) ) {
            if ( class_exists( 'WP_CLI' ) ) {
                \WP_CLI::error( 'Vercel deploy hook is not defined.' );
            }

            return;
        }
    
        $transient_name = 'vercel_deployment_status';
    
        $response = wp_remote_get( VERCEL_DEPLOY_HOOK );
        $body = json_decode( wp_remote_retrieve_body( $response ), true );
    
        if ( is_array( $body ) && $body['job']['state'] === 'PENDING' ) {
            set_transient( $transient_name, 'INITIALIZING', DAY_IN_SECONDS );

            if ( class_exists( 'WP_CLI' ) ) {
                \WP_CLI::success( 'Vercel Deployment initializing' );
            }
        }
    }


    /**
     * Get the latest deployment status from Vercel.
     */
    static function vercel_get_deployment_status() {
        wp_send_json([
            'status' => get_transient( 'vercel_deployment_status' )
        ]);
    }


    /**
     * Set the status for the latest deployment on Vercel.
     */
    static function vercel_set_deployment_status() {
        $status = $_REQUEST['deployment_status'];
    
        if ( $status === 'ERROR' ) {
            $expiration = MONTH_IN_SECONDS;
        } else {
            $expiration = DAY_IN_SECONDS;
        }
    
        wp_send_json([
            'success' => set_transient( 'vercel_deployment_status', $status, $expiration ),
            'status' => $status,
            'expiration' => $expiration
        ]);
    }
}

new Nextjs;
